/**********************************************************************************
 * Project: G2HDM - 2 Higgs Doublet fitting package                               *
 * Package: G2HDM                                                                 *
 * Class  : Delta_Rb                                                              *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * Sources:                                                                       *
 *     (1)   Howard E. Haber, Heather E. Logan, Phys.Rev. D62 (2000) 015011       *
 *             hep-ph/9909335                                                     *
 *     (2) some results taken from J. Field, Mod. Phys. Lett. A13, 1937 (1998)    *
 *                                                                                *
 * see corresponding .h file for author and license information                   *            
 *                                                                                *
 **********************************************************************************/

#include <cmath>
#include <iostream>

#include "TMath.h"

#include "Gfitter/GParameterRef.h"
#include "Gfitter/GMath.h"
#include "Gfitter/GConstants.h"

//#include "Gfitter/GVariable.h"
//#include "Gfitter/GStore.h"

#include "G2HDM/Delta_Rb.h"

using namespace Gfitter;

ClassImp(G2HDM::Delta_Rb)

G2HDM::Delta_Rb::Delta_Rb() 
   : G2HDM::G2HDMBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
   BookParameter( "MZ",      & p_MZ );
   BookParameter( "mt",      & p_mt );
   BookParameter( "MW",      & p_MW );
        
   BookTheory   ( "GEW::QMassRunning",   &t_qMassRun );
}

Double_t G2HDM::Delta_Rb::GetTheoryPrediction() 
{ 
   // Reference:  hep-ph/9909335  H.E. Haber and H.E. Logan    (1)
   // some results taken from J. Field, Mod. Phys. Lett. A13, 1937 (1998)    (2)
   Double_t tanBeta     = GetTanb();
   Double_t mChargedH   = GetMHch();
   
   Double_t vev = GConstants::vev();
   Double_t g   = 2*p_MW/vev;

   Double_t mbatMZ = GetQMassRun().GetRunningQuarkMass( GTypes::kBottom );

   Double_t RtH = std::pow(p_mt/mChargedH,2);
   Double_t Rfactor = RtH/(RtH-1) - RtH*TMath::Log(RtH)/(std::pow(RtH-1,2));

   // --- e/swcw should not be there, because it is factored out in the 
   // --- definition of g_L and g_R, see eqs.(2.1) and (2.2) of HL
   // --- thanks to Degrassi and Slavich for pointuing this out!
   //Double_t costh = p_MW/p_MZ;
   //Double_t sinth = std::sqrt(1 - std::pow(p_MW/p_MZ,2));
   //Double_t ef  = g*sinth*sinth; 
   //Double_t prefactor = ef/(32*std::pow(TMath::Pi(),2)*sinth*costh);
   Double_t prefactor = 1/(32*std::pow(TMath::Pi(),2));
   
   Double_t Lfac= g*p_mt/(std::sqrt(2)*p_MW*tanBeta);
   Double_t Rfac= g*mbatMZ*tanBeta/(std::sqrt(2)*p_MW);


   Double_t dgbL2hdm = prefactor*Lfac*Lfac*Rfactor; 
   Double_t dgbR2hdm = -prefactor*Rfac*Rfac*Rfactor;

   Double_t DeltaRb = -0.7785*dgbL2hdm + 0.1409*dgbR2hdm;
  
   return DeltaRb;
}

