/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : DAlphaQED                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory of the Delta-Alpha-QED at the Z-pole                     *
 *      leptonic contribution to effective em coupling constant                   *
 *      up to three loops                                                         *
 *                                                                                *
 * Sources:                                                                       *
 *       (i)  M. Steinhauser                                                      *
 *            Phys. Lett. B 429, 158 (1998), [hep-ph/9803313]                     *
 *       (ii) J. H. Kuhn and M. Steinhauser                                       *
 *            Phys. Lett. B 437, 425 (1998), [hep-ph/9802241]                     *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GConstants.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"

#include "GEW/DAlphaQED.h"
#include "GEW/RunningAlphaQCD.h"

#include "Riostream.h"

using namespace Gfitter;
using std::complex;

ClassImp(GEW::DAlphaQED)

GEW::DAlphaQED::DAlphaQED()
   : Gfitter::GAuxTheory(),
     m_isUpToDate_Update( kFALSE )     
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   BookParameter( "MZ"        , &p_MZ );
   BookParameter( "mt"        , &p_mt );
   BookParameter( "DAlphaHad" , &p_DAlphaHad );
    
   BookTheory   ( "GEW::RunningAlphaQCD" , &t_AlphasRun );
}

void GEW::DAlphaQED::Initialise()
{
   // lepton masses
   m_mLep2[0] = GMath::IPow( GConstants::me()  , 2 );
   m_mLep2[1] = GMath::IPow( GConstants::mmu() , 2 );
   m_mLep2[2] = GMath::IPow( GConstants::mtau(), 2 );
}

void GEW::DAlphaQED::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

// Update parmeters and check if something has changed
void GEW::DAlphaQED::Update()
{
   if (m_isUpToDate_Update) return;
  
   // now, it is uptodate (I mean... it will be)
   m_isUpToDate_Update = kTRUE;

  
   Double_t DAlphaLept = DAlphaLep(p_MZ*p_MZ);
  
   // excluding top contribution
   m_DAlphaQED    = DAlphaLept + p_DAlphaHad;  
  
   // including top contribution
   m_DAlphaQEDtop = DAlphaLept + p_DAlphaHad + DAlphatop(p_MZ*p_MZ) ;  

   // protection against unphysical values
   if (TMath::Abs(p_MZ)<10e-6 || TMath::Abs(p_MZ)>10e6 ||
       TMath::Abs(p_mt)<10e-6 || TMath::Abs(p_mt)>10e6 ||
       TMath::Abs(p_DAlphaHad)<10e-6 || TMath::Abs(p_DAlphaHad)>10e6 ){
     m_DAlphaQED=0.;
     m_DAlphaQEDtop=0.;
   }
   SetUpToDate();
}

// leptonic contribution
// eq. (1) from (i), incl sub-equation (5) - (10)
// note: DeltaAlphaLep = -Pi(MZ^2)
Double_t GEW::DAlphaQED::DAlphaLep( Double_t s )
{

   // protection against unphysical values
   if (TMath::Abs(s)<10e-6 || TMath::Abs(s>10e12)){
     return 0.;
   }

   Double_t Pi0 = 0;
   Double_t Pi1 = 0;
   Double_t Pi2A = 0;
   Double_t Pi2l = 0;
   Double_t Pi2F = 0;
   Double_t Pi2h = 0;
    
   for (Int_t i = 0; i < 3; i++) {
      Double_t Ratio = m_mLep2[i]/s;
      Double_t logRatio  = TMath::Log( 1./Ratio );
      
      // Pi(0) eq. (5) 
      Pi0 += 20/9.0 - 4/3.0*logRatio + 8.0*Ratio;
      
      // Pi(1) eq. (6)
      Pi1 += 5/6.0 - 4*GMath::Zeta3() - logRatio - 12.0*Ratio*logRatio;
      
      // Pi(2)A eq. (7)
      Pi2A += -121/48.0 + (-5.0 + 8.0*TMath::Log(2.0))*GMath::Zeta2()
         -99/16.0*GMath::Zeta3() + 10.0*GMath::Zeta5() + 1/8.0*logRatio ;
      
      // Pi(2)F eq. (9)
      Pi2F += -307/216.0 - 8/3.0*GMath::Zeta2() + 545/144.0*GMath::Zeta3()
         +(11/6.0 - 4/3.0*GMath::Zeta3())*logRatio - 1/6.0*logRatio*logRatio;
      
      for (Int_t j = 0; j < 3; j++) {
         Double_t logM1 = TMath::Log( s/m_mLep2[i] );
         Double_t logM2 = TMath::Log( s/m_mLep2[j] );
            
         if( i > j ) {
            // Pi(2)l eq. (8)
            // for electrons (i=0) term not present
            // for muons (i=1) j=0 (electron)
            // for taus (i=2) j=0,1 (electron and muon)
            Pi2l += -116/27.0 + 4/3.0*GMath::Zeta2() + 38/9.0*GMath::Zeta3() 
               + 14/9.0*logM1 + (5/18.0 - 4/3.0*GMath::Zeta3())*logM2
               + 1/6.0*logM1*logM1 - 1/3.0*logM1*logM2;
         }
         else if( j > i ) {
            // Pi(2)h eq. (10) (treatment vice versa to Pi(2)l)
            // for electrons (i=0) j=1,2 (muon and tau)
            // for muons (i=1) j=2 (tau)
            // for taus (i=2) not present
            Pi2h += -37/6.0 + 38/9.0*GMath::Zeta3() + (11/6.0 - 4/3.0*GMath::Zeta3())*logM2 
               - 1/6.0*logM2*logM2;
         }
      }      
   }
   
   //eq. (1) from (i), not Delta_alpha = -Pi(q^2)
   double dalpha = -GConstants::alphaQED()/(4.*TMath::Pi())*
      ( Pi0 + GConstants::alphaQED()/TMath::Pi()*Pi1 
        + GMath::IPow(GConstants::alphaQED()/TMath::Pi(), 2)*(Pi2A+Pi2l+Pi2F+Pi2h) );   

   return dalpha;
}


// eq. (12) from (ii)
Double_t GEW::DAlphaQED::DAlphatop( Double_t s )
{

   // protection against unphysical values
   if (TMath::Abs(s)<10e-6 || TMath::Abs(s>10e12)){
     return 0.;
   }

   Double_t As        = GetAlphasRun().EvolveAlphas( TMath::Sqrt(s) )/TMath::Pi();
   Double_t logMu2Mt2 = TMath::Log((p_MZ*p_MZ)/(p_mt*p_mt));

   Double_t atop = ( -4/45.0*GConstants::alphaQED()/TMath::Pi()*s/(p_mt*p_mt)*
                     (1.0 + 5.062*As + (28.220 + 9.702*logMu2Mt2)*As*As
                      + s/(p_mt*p_mt)*(0.1071 + 0.8315*As + (6.924 + 1.594*logMu2Mt2)*As*As)) );
   return atop;
}
      
