/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : GammaZlept                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Prediction for partial width of Z0->ll (usually not used in fit)          *
 *                                                                                *
 * Source:                                                                        *
 *      (i) A. Ferroglia, G. Ossola, M. Passera and A. Sirlin                     *
 *          Phys. Rev.  D 65, 113002 (2002), [arXiv:hep-ph/0203224]               *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GEW/AlphaQCDAtQ.h"
#include "GEW/GammaZlept.h"
#include "GEW/MH.h"

using namespace Gfitter;

ClassImp(GEW::GammaZlept)

GEW::GammaZlept::GammaZlept()
   : Gfitter::GTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
  
   // theory choice
   const TString& omstype = gStore()->GetVariable( "GEWFlags::OMSType" )->GetStringValue();
   m_logger << kINFO << "Using OMs type: \"" << omstype << "\"" << GEndl;
   
   if      (omstype == "OMS1" ){
      m_OMSType  = OMS1;
   }
   else if (omstype == "OMS1_2"){
      m_OMSType  = OMS1_2;
   }
   else if (omstype == "OMS2"){
      m_OMSType  = OMS2;
   }
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::OMStype\": \"" << omstype << "\""
               << ". Possible are: \"OMS1\", \"OMS1-2\" \"OMS2\""
               << GEndl;     
   }
  
   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }
   
   BookParameter( "mt",             &p_mt );   
   BookParameter( "DAlphaHad",      &p_DAlphaHad );    
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );
   BookTheory   ( "GEW::MH" , & t_MH );
}

// coefficients from table 3 of (i) 
void GEW::GammaZlept::Initialise()
{
   switch(m_OMSType){
   case OMS1:
   case OMS1_2: 
      m_gZ0 = 84.0174;
      m_g1  = 0.04472;
      m_g2  = 0.476;
      m_g3  = 0.804;
      m_g4  = 0.113;
      m_g5  = 0.03673;
      break;
   case OMS2:
      m_gZ0 = 84.0152;
      m_g1  = 0.04568;
      m_g2  = 0.476;
      m_g3  = 0.801;
      m_g4  = 0.112;
      m_g5  = 0.03683;
      break;
   }
}

// eq. (15) from (i)
Double_t GEW::GammaZlept::GetTheoryPrediction()
{
   Double_t MH      = GetMH().GetValue();
   if( m_logMH ) MH = TMath::Exp( GetMH().GetValue() ); 
   
   Double_t A1  = TMath::Log(MH/100.0);
   Double_t A2  = (p_DAlphaHad/0.02761) - 1.0;
   Double_t A3  = ((p_mt/174.3)*(p_mt/174.3) - 1.0);
   Double_t A4  = (GetAlphasMZ()/0.118) - 1.0;   
     
   Double_t gzlep  = ( m_gZ0 - m_g1*A1 - m_g5*A1*A1 - m_g2*A2 + m_g3*A3 - m_g4*A4 );

   return gzlep;
}
