/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : HiggsStabilityBound                                                   *
 *                                                                                *
 * Description:                                                                   *
 *      Bound obtained from vacuum stability requirement:                         *
 *      lambda(mu) > 0 for mu < Lambda_cutoff                                     *
 *                                                                                *
 * Sources:                                                                       *
 *      (i)  J. Ellis, J. R. Espinosa, G. F. Giudice, A. Hoecker and A. Riotto    *
 *           Phys. Lett. B 679, 369 (2009), [arXiv:0906.0954 [hep-ph]]            *
 *      (ii) J. R. Espinosa, G. F. Giudice and A. Riotto                          *
 *           JCAP 0805, 002 (2008), [arXiv:0710.2484 [hep-ph]]                    *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
**********************************************************************************/

#include "TMath.h"
#include "Riostream.h"
#include "TGraph.h"

#include "Gfitter/GMath.h"

#include "GEW/HiggsStabilityBound.h"

#define UseFiles__ kTRUE

// reference values used in interpolation
#define MTOP_REF__ 173.1
#define MTOP_SIG__ 1.3
#define ASMZ_REF__ 0.1193
#define ASMZ_SIG__ 0.0028

using namespace Gfitter;

ClassImp(GEW::HiggsStabilityBound)

GEW::HiggsStabilityBound::HiggsStabilityBound() 
   : HiggsTheoryBound(),
     m_sb_ref(0), 
     m_sb_mtp(0), 
     m_sb_mtm(0),  
     m_sb_asp(0),  
     m_sb_asm(0) 
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   BookParameter( "MHcStabilityError",   &p_criticalHiggsMassError );         
   BookParameter( "mt",                  &p_mt );   

   BookTheory   ( "GEW::AlphaQCDAtQ/MZ", &t_AlphasMZ ); 
}

void GEW::HiggsStabilityBound::Initialise()
{
   m_boundType = kLower;
   HiggsTheoryBound::Initialise();

   if (UseFiles__) {
      // Reference: (ii) - data files by Jose Espinosa, U. Penn
      m_sb_ref = GetGraphFromFile( "datafiles/sb_ref.dat" );
      m_sb_mtp = GetGraphFromFile( "datafiles/sb_mt_p.dat" );
      m_sb_mtm = GetGraphFromFile( "datafiles/sb_mt_m.dat" );
      m_sb_asp = GetGraphFromFile( "datafiles/sb_as_p.dat" );
      m_sb_asm = GetGraphFromFile( "datafiles/sb_as_m.dat" );
   }
}

Double_t GEW::HiggsStabilityBound::GetCriticalMass()
{
   // units in GeV
   Double_t MHc = 0;

   if (UseFiles__) {
      // Reference: (ii) - data files by Jose Espinosa, U. Penn

      Double_t alphasMZ = GetAlphasMZ();

      // positive or negative deviations?
      const TGraph* g_as = (alphasMZ > ASMZ_REF__ ? m_sb_asp : m_sb_asm);
      const TGraph* g_mt = (p_mt     > MTOP_REF__ ? m_sb_mtp : m_sb_mtm);

      // get reference interpolations
      Double_t MHc_ref = m_sb_ref->Eval( p_Log10LambdaCutoff );     
      Double_t MHc_mt  = g_mt    ->Eval( p_Log10LambdaCutoff );
      Double_t MHc_as  = g_as    ->Eval( p_Log10LambdaCutoff );
      
      // compute MH
      MHc = ( MHc_ref 
              + TMath::Abs(p_mt     - MTOP_REF__)/MTOP_SIG__ * (MHc_mt - MHc_ref)
              + TMath::Abs(alphasMZ - ASMZ_REF__)/ASMZ_SIG__ * (MHc_as - MHc_ref) );  

      // add theoretical error
      MHc += p_criticalHiggsMassError;
   }
   else {
      // eq. (4) from (i)
      MHc = ( 128.6 + 2.6*(p_mt - 173.1)/1.3 - 2.2*(GetAlphasMZ() - 0.1193)/0.0028
              + p_criticalHiggsMassError );
   }
   
   return MHc;
}
