/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : HiggsTheoryBound                                                      *
 *                                                                                *
 * Description:                                                                   *
 *      This base class implements strict (!) bounds on the Higgs boson mass,     *
 *      where strict means MH >(<) MH_critical gets a infinite contribution       *
 *      to the chi-squared                                                        *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
**********************************************************************************/

#include "TMath.h"
#include "TGraph.h"
#include "TTree.h"
#include "TFile.h"
#include "Riostream.h"
#include "TSystem.h"

#include "GEW/HiggsTheoryBound.h"
#include "GEW/MH.h"

ClassImp(GEW::HiggsTheoryBound)

using namespace Gfitter;

GEW::HiggsTheoryBound::HiggsTheoryBound() 
   : Gfitter::GTheory(),
     m_boundType( kNone ),
     m_sign     ( 0 )
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
   
   BookParameter( "Log10LambdaCutoff", &p_Log10LambdaCutoff );      
   BookTheory   ( "GEW::MH" , & t_MH );
}

void GEW::HiggsTheoryBound::Initialise()
{
   if (m_boundType == kNone) {
      m_logger << kFATAL << "Did not set bound type" << GEndl;
   }
   m_sign = m_boundType == kUpper ? -1 : +1;
}

void GEW::HiggsTheoryBound::CalculateVariables()
{
   // nothing so far
}

Double_t GEW::HiggsTheoryBound::GetTheoryPrediction()
{
   CalculateVariables();
   Double_t MHc = GetCriticalMass();
  
   return ( GetMH().GetValue() - MHc)*m_sign;
}

const TGraph* GEW::HiggsTheoryBound::GetGraphFromFile( const TString& fname ) const
{
   m_logger << kINFO << "<HiggsTheoryBound::GetGraphFromFile> Creating TGraph from file: " << fname << GEndl;

   TString f = fname;
   f.Replace(0, fname.Index('/')+1, "");
   f.ReplaceAll(".dat","");  
   f.ReplaceAll(".txt","");

   // create graph from file
   TGraph* graph = new TGraph( fname );
   if (!graph) {
      m_logger << kFATAL << "Zero TGraph pointer for file: " << fname << GEndl;
   }

   // graph must be sorted in X
   graph->Sort();

   // write to file for inspection
   TString g = f + "_graph";
   graph->SetNameTitle( g, g );
   gStore()->GetTargetRootFile()->cd();
   graph->Write( g );

   //   delete tree;

   return graph;
}
