/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : HiggsTrivialityBound                                                  *
 *                                                                                *
 * Description:                                                                   *
 *      Bound obtained from triviality requirement (Landau pole)                  *
 *                                                                                *
 * Sources:                                                                       *
 *      (i)  J. Ellis, J. R. Espinosa, G. F. Giudice, A. Hoecker and A. Riotto    *
 *           Phys. Lett. B 679, 369 (2009), [arXiv:0906.0954 [hep-ph]]            *
 *      (ii) J. R. Espinosa, G. F. Giudice and A. Riotto                          *
 *           JCAP 0805, 002 (2008), [arXiv:0710.2484 [hep-ph]]                    *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
**********************************************************************************/

#include "TMath.h"
#include "Riostream.h"

#include "Gfitter/GMath.h"
#include "TGraph.h"

#include "GEW/HiggsTrivialityBound.h"

#define UseHighBound__ kTRUE

using namespace Gfitter;

ClassImp(GEW::HiggsTrivialityBound)

GEW::HiggsTrivialityBound::HiggsTrivialityBound() 
   : HiggsTheoryBound(),
     m_pb_ref(0)
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   BookParameter( "mt",                  &p_mt );   
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ", &t_AlphasMZ ); 
}

void GEW::HiggsTrivialityBound::Initialise()
{
   m_boundType = kUpper;
   HiggsTheoryBound::Initialise();

   // Reference: (ii) - data files by Jose Espinosa, U. Penn
   m_pb_ref = GetGraphFromFile( (UseHighBound__ ? "datafiles/pbH.dat" : "datafiles/pbL.dat") );
}

Double_t GEW::HiggsTrivialityBound::GetCriticalMass()
{
   // units in GeV
   Double_t MHc = 0;

   // Reference: (ii) - data files by Jose Espinosa, U. Penn
   MHc = m_pb_ref->Eval( p_Log10LambdaCutoff );  

   // Eq (3) from (i)
   Double_t as = GetAlphasMZ();
   MHc += ( 0.7*(p_mt - 173.1)/1.3 - 0.4*(as - 0.1193)/0.0028 );      

   return MHc;
}
