/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : QMassRunning                                                          *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory for running quark masses ( only charms and bottoms )     *
 *                                                                                *
 * Source:                                                                        *
 *     (i) J. A. M. Vermaseren, S. A. Larin and T. van Ritbergen                  *
 *         Phys. Lett.  B 405, 327 (1997), [arXiv:hep-ph/9703284]                 *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GEW/QMassRunning.h"
#include "GEW/RunningAlphaQCD.h"
#include "GEW/AlphaQCDAtQ.h"

using namespace Gfitter;

ClassImp(GEW::QMassRunning)

GEW::QMassRunning::QMassRunning()
   : GAuxTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
   
   BookParameter( "mc_MSb"       , &p_mc );   
   BookParameter( "mb_MSb"       , &p_mb ); 

   BookTheory( "GEW::AlphaQCDAtQ/MZ",       & t_AlphasMZ );  
   BookTheory( "GEW::RunningAlphaQCD" , & t_AlphasRun );
}

// compute running mass in the MS_bar scheme 
Double_t GEW::QMassRunning::MSMassRunning( GTypes::Particle ParticleType )
{
   Double_t qm_hat = 0;
   
   switch( ParticleType ){
   case GTypes::kCharm:
      qm_hat = p_mc / EvolveMass( 1.0, GetAlphasRun().EvolveAlphas(p_mc), ParticleType );
      break;
   case GTypes::kBottom:
      qm_hat = p_mb / EvolveMass( 1.0, GetAlphasRun().EvolveAlphas(p_mb), ParticleType );
      break;
   default:
      m_logger << kFATAL << "<MSMassRunning> flavour not supported: " << GTypes::GetName(ParticleType) << GEndl;
   } 
   return EvolveMass( qm_hat, GetAlphasMZ(), ParticleType);;
}

// evolve quark mass running in MS_bar scheme
// eq. (20) from (i)
Double_t GEW::QMassRunning::EvolveMass( Double_t m, Double_t alphas, GTypes::Particle ParticleType )
{
   Double_t RunMass = 0;
   Double_t As      = alphas/TMath::Pi();  

   switch( ParticleType ){
   case GTypes::kCharm:
      RunMass  = ( m*std::pow(As,12/25.0)
                   *( 1.0 + 1.01413*As + 1.38921*As*As + 1.09054*As*As*As ) );
      break;
   case GTypes::kBottom:
      RunMass  = ( m*std::pow(As,12/23.0)
                   *( 1.0 + 1.17549*As + 1.50071*As*As + 0.172478*As*As*As ) );
      break;
   default:
      m_logger << kFATAL << "<EvolveMass> flavour not supported: " << GTypes::GetName(ParticleType) << GEndl;
   } 
   return RunMass;
}


