/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : QPoleMass                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory for inverting Ms-bar masses to pole quark masses         *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/
#ifndef GEW_QPoleMass
#define GEW_QPoleMass

#include "Gfitter/GTypes.h"
#include "Gfitter/GAuxTheory.h"




using namespace Gfitter;

namespace GEW {

   class RunningAlphaQCD;

   class QPoleMass : public Gfitter::GAuxTheory {

   public:
      
      QPoleMass();
      ~QPoleMass() {}

      // not used in Aux Theories
      Double_t GetTheoryPrediction() { return 0; }
    
      // accessor; downcast to use it
      Double_t GetPoleQMass( GTypes::Particle );
         
      // not used
      void Initialise() {}

   protected:

      Double_t MSMassToPoleMass( Double_t m , Int_t nf );
      
      // reference to AlphasRunning
      RunningAlphaQCD& GetAlphasRun() const { return (RunningAlphaQCD&)*t_AlphasRun; }
      Gfitter::GTheoryRef    t_AlphasRun; 

      // reference to AlphasMZ
      Gfitter::GTheory& GetAlphasMZ() const { return (Gfitter::GTheory&)*t_AlphasMZ; }   
      Gfitter::GTheoryRef    t_AlphasMZ; 
    
      // quark masses (at pole)
      Gfitter::GParameterRef p_mu; 
      Gfitter::GParameterRef p_md; 
      Gfitter::GParameterRef p_ms; 
      Gfitter::GParameterRef p_mc; 
      Gfitter::GParameterRef p_mb; 
 
      ClassDef(QPoleMass,0)

   };
}

// quark pole masses, computed from MS_bar masses
inline Double_t GEW::QPoleMass::GetPoleQMass( GTypes::Particle part )
{
   switch (part) {
   case GTypes::kUp:
      return MSMassToPoleMass( p_mu, 1 ); 
   case GTypes::kDown:
      return MSMassToPoleMass( p_md, 2 ); 
   case GTypes::kStrange:
      return MSMassToPoleMass( p_ms, 3 ); 
   case GTypes::kCharm:
      return MSMassToPoleMass( p_mc, 4 ); 
   case GTypes::kBottom:
      return MSMassToPoleMass( p_mb, 5 ); 
   default:
      m_logger << kFATAL 
               << "<GetPoleQMass> Unknown quark flavour: " << part << GEndl;
   } 

   return 0;
}

#endif
