/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : R0bParametrisation                                                    *
 *                                                                                *
 * Description:                                                                   *
 *       Ratio of the bottoms partial Z width to all hadrons (full two loop)      *
 *                                                                                *
 * Papers:                                                                        *
 *     (i)  A. Freitas and Yi-Cheng Huang                                         *
 *          arXiv: 1205.0299v1 [hep-ph]                                           *
 *          (need to be confirmed be final publication)                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GConstants.h"

#include "GEW/R0bParametrisation.h"

using namespace Gfitter;

ClassImp(GEW::R0bParametrisation)

GEW::R0bParametrisation::R0bParametrisation() : 
   Gfitter::GAuxTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }
   
   BookParameter( "MZ",                   &p_MZ );   
   BookParameter( "mt",                   &p_mt );   
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );   
   BookTheory   ( "GEW::DAlphaQED",       &t_DAlphaQED );   
   BookTheory   ( "GEW::MH",              &t_MH );
}

// initialise coefficienct
// eq. (23) of (i) 
void GEW::R0bParametrisation::Initialise()
{
   // m_r0b =  0.2147464;
   // m_c1  =  2.21e-5;
   // m_c2  =  2.6e-6;
   // m_c3  = -6.7e-7;
   // m_c4  =  9.11e-8;
   // m_c5  =  6.47e-4;
   // m_c6  = -3.239e-3;
   // m_c7  =  6.73e-5;
   // m_c8  = -3.24e-4;
   // m_c9  =  6.10e-2;

   m_r0b =  0.2149246;
   m_c1  =  2.23e-5;
   m_c2  =  2.6e-6;
   m_c3  = -6.8e-7;
   m_c4  =  9.19e-8;
   m_c5  =  6.58e-4;
   m_c6  = -3.363e-3;
   m_c7  =  6.74e-5;
   m_c8  = -1.688e-3;
   m_c9  = -9.26e-4;
   m_c10 =  5.93e-2;

}


// paramtrized formula of (i), see eqs. (21) and (22)
Double_t GEW::R0bParametrisation::GetR0b()
{   

   Double_t r0b = 0;

   Double_t MH      = TMath::Abs( GetMH().GetValue() );
   if( m_logMH ) MH = TMath::Exp( GetMH().GetValue() ); 
      
   Double_t LogH        = TMath::Log(MH/100.);
   Double_t dH          = MH/100.;

   Double_t dt          = (p_mt/173.2)*(p_mt/173.2) - 1.0;
   Double_t dZ          = (p_MZ/91.1876) - 1.0;

   Double_t deAlpha     = (GetDAlphaQED().DAlphaQEDMZ()/0.05900) - 1.0;
   Double_t deAlphaS    = (GetAlphasMZ()/0.1184) - 1.0;   


   r0b = m_r0b + m_c1*LogH + m_c2*LogH*LogH + m_c3*LogH*LogH*LogH*LogH + m_c4*(dH*dH-1) + m_c5*deAlpha 
     + m_c6*dt + m_c7*dt*LogH + m_c8*deAlphaS + m_c9*deAlphaS*deAlphaS + m_c10*dZ;

   //using namespace std;
   //cout << "R0b: " << " MH = " << MH << " mt = " << p_mt << " MZ = " << p_MZ << " alpha = " << GetDAlphaQED().DAlphaQEDMZ()
   // << " alphaS = " << (deAlphaS + 1.0) * 0.1184 << endl;
   //cout << "R0b = " << r0b << endl << endl;  

   return r0b;
}
