/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : RunningAlphaQCD                                                       *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory for running coupling constant                            *
 *                                                                                *
 * Sources:                                                                       *
 *      (i)   G. Rodrigo, A. Pich and A. Santamaria                               *
 *            Phys. Lett. B 424, 367 (1998), [arXiv:hep-ph/9707474]               *
 *      (ii)  G. M. Prosperi, M. Raciti and C. Simolo                             *
 *            Prog. Part. Nucl. Phys. 58, 387 (2007), [arXiv:hep-ph/0607209]      *
 *      (iii) B. A. Magradze                                                      *
 *            Few Body Syst. 40, 71 (2006), [arXiv:hep-ph/0512374]                *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
 **********************************************************************************/
#include "TMath.h"
#include "Riostream.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GRungeKutta.h"
#include "Gfitter/GRootFinder.h"

#include "GEW/RunningAlphaQCD.h"
#include "GEW/AlphaQCDAtQ.h"

using namespace Gfitter;

ClassImp(GEW::RunningAlphaQCD)

GEW::RunningAlphaQCD:: RunningAlphaQCD()
   : Gfitter::GAuxTheory(),
   m_isUpToDate_Update( kFALSE )  
{
   const TString& AlphasType = gStore()->GetVariable( "GEWFlags::OptionQCD" )->GetStringValue();
   m_logger << kINFO << "Evaluate running alpha_s(s) with: \"" << AlphasType << "\" method" << GEndl;

   m_thisPointer = this;
  
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
  
   if (AlphasType == "FitLambda" ){
      BookParameter( "lambdaMS5", &p_InputValueLambdaMS5 );
      m_QCDType  = kFitLambda;
   }
   else if (AlphasType == "FitAlphas"){
      BookParameter("alphasMZ", &p_InputValueAlphasMZ );
      m_QCDType  = kFitAlphas;
   }
   else if (AlphasType == "RungeKutta"){
      m_logger << kINFO << "Booking alphasMZ" << GEndl;
      BookParameter("alphasMZ", &p_InputValueAlphasMZ );
      m_QCDType  = kRungeKutta;
   }  
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::OptionQCD\": \"" 
               << AlphasType << "\"" << ". Possible are: \"FitLambda\", \"FitAlphas\", \"RungeKutta\""
               << GEndl;
   }
   
   BookParameter( "MZ"        , &p_MZ );
   BookParameter( "mu_MSb"    , &p_mu );
   BookParameter( "md_MSb"    , &p_md );
   BookParameter( "ms_MSb"    , &p_ms );
   BookParameter( "mc_MSb"    , &p_mc );
   BookParameter( "mb_MSb"    , &p_mb );
   BookParameter( "mt"        , &p_mt );
   BookParameter( "Scale"     , &p_Scale );
}

void GEW::RunningAlphaQCD::UpdateLocalFlags( GReference& /* ref */ )
{}

void GEW::RunningAlphaQCD::Update()
{

   if (m_isUpToDate_Update) return;

} 

void GEW::RunningAlphaQCD::Initialise()
{
   Double_t pi = TMath::Pi();

   // init coefficients for alphas matching 
   // eq. (8) from (i)
   m_c20 = -11/72.0;
   
  // eq. (8) from (i)
   for (Int_t nf = 0; nf < kNfmax; nf++) {
      m_c30[nf] = ( 82043/27648.0*GMath::Zeta3() 
                    - 575263/124416.0 + 2633/31104.0*nf );
   }
   // init "beta" functions
   // eq. (16) from (ii)
   for (Int_t nf = 0; nf < kNfmax; nf++) {
      m_b0[nf] = 1/(4.0*pi)*(11.0 - 2/3.0*nf);
      m_b1[nf] = 1/GMath::IPow(4.0*pi,2)*(102.0 - 38/3.0*nf);
      m_b2[nf] = 1/GMath::IPow(4.0*pi,3)*(2857/2.0 - 5033/18.0*nf + 325/54.0*nf*nf);
      m_b3[nf] = ( 1/GMath::IPow(4.0*pi,4)*
                   ( (149753/6.0 + 3564.0*GMath::Zeta3())
                     - (1078361/162.0 + 6508/27.0*GMath::Zeta3())*nf
                     + (50065/162.0 + 6472/81.0*GMath::Zeta3())*nf*nf 
                     + 1093/729.0*nf*nf*nf ) );
   }
}

// ---- switching between different methods for alpha_s running ---
// ---- Fit methods or Runge-Kutta integration ---
Double_t GEW::RunningAlphaQCD::EvolveAlphas( Double_t mu )
{   
   if (mu == p_MZ) {
      if (GetQCDType() == kFitAlphas || GetQCDType() == kRungeKutta ) { 
         // in this case just return the parameter (ie. a_s(M_z) from data card)
         return p_InputValueAlphasMZ;   
      }
      else {
         return AlphaStrong( 5, mu, p_InputValueLambdaMS5 );
      }
   }
   
   Double_t asnew  = 0; 
   Int_t    nf     = 0;
   
   //calculate number of active flavours
   
   if (mu < 1 || TMath::IsNaN(mu) ){
     m_logger << kWARNING << " <EvolveAlphas>: Energy-Scale is too low ( scale < 1 GeV ) or NaN. Scale = "
	      << mu << GEndl;
     return asnew;
   }
   else    nf = 3;
      
   if (mu >= p_mc) nf = 4;
   if (mu >= p_mb) nf = 5;
   if (mu >= p_mt) nf = 6;

   if( GetQCDType() == kFitAlphas || GetQCDType() == kFitLambda ) {
      Double_t lambda = 0;      
      switch(nf) {
      case 3:
         if (TMath::IsNaN(p_mc)) m_logger << kFATAL << "<RunningAlphaQCD::EvolveAlphas> p_mc is NaN !" << GEndl;
         asnew  = EvolveAlphas( p_mc );  
         asnew  = AlphasMatchDown( 4, asnew, p_Scale*p_mc, p_mc );
         lambda = LambdaAtThreshold ( 3, asnew, p_Scale*p_ms );
         asnew  = AlphaStrong ( 3, mu, lambda );  
         break;
      case 4:
         asnew  = AlphaStrong( 5, p_Scale*p_mb, GetLambdaMS5() );
         asnew  = AlphasMatchDown( 5, asnew, p_Scale*p_mb, p_mb );
         lambda = LambdaAtThreshold   (    4, asnew, p_Scale*p_mb );
         asnew  = AlphaStrong( 4, mu, lambda );  
         break;
      case 5:
         asnew  = AlphaStrong( 5, mu, GetLambdaMS5() );
         break;
      case 6:
         asnew  = AlphaStrong(  5, p_Scale*p_mt, GetLambdaMS5() );
         asnew  = AlphasMatchUp( 6, asnew, p_Scale*p_mt, p_mt );
         lambda = LambdaAtThreshold   (  6, asnew, p_Scale*p_mt );
         asnew  = AlphaStrong( 6, mu, lambda );
         break;
      }
   }
   else if( GetQCDType() == kRungeKutta ) {
      if (nf == 5) {
         GRungeKutta rungekutta5( &GetRGE5 );
         asnew = rungekutta5.GetYEndValue( 2.0*TMath::Log( p_MZ ), p_InputValueAlphasMZ,
                                           2.0*TMath::Log( mu ) );
      } 
      else if( nf == 4 ) {
         if (TMath::IsNaN(p_mb)) m_logger << kFATAL << "<RunningAlphaQCD::EvolveAlphas> p_mb is NaN !" << GEndl;
         asnew = EvolveAlphas( p_mb );  
         asnew = AlphasMatchDown( 5, asnew, p_Scale*p_mb, p_mb );
         GRungeKutta rungekutta4( &GetRGE4 );
         asnew = rungekutta4.GetYEndValue( 2.0*TMath::Log( p_mb ), asnew,
                                           2.0*TMath::Log( mu ) ); 
      }
      else if( nf == 3 ) {
         if (TMath::IsNaN(p_mc)) m_logger << kFATAL << "<RunningAlphaQCD::EvolveAlphas> p_mc(s) is NaN !" << GEndl;
         asnew  = EvolveAlphas( p_mc );
         asnew  = AlphasMatchDown( 4, asnew, p_Scale*p_mc, p_mc );
         GRungeKutta rungekutta3( &GetRGE3 );
         asnew = rungekutta3.GetYEndValue( 2.0*TMath::Log( p_ms ), asnew,
                                           2.0*TMath::Log( mu ) ); 
      }
      else if (nf == 6 ) {
         GRungeKutta rungekutta5( &GetRGE5 );
         asnew = rungekutta5.GetYEndValue( 2.0*TMath::Log( p_MZ ), p_InputValueAlphasMZ,
                                           2.0*TMath::Log( p_mt ) ); 
         asnew  = AlphasMatchUp( 6, asnew, p_Scale*p_mt, p_mt );
         GRungeKutta rungekutta6( &GetRGE6 );
         asnew = rungekutta6.GetYEndValue( 2.0*TMath::Log( p_mt ), asnew, 
                                           2.0*TMath::Log( mu ) ); 
      }
   }
    
   return asnew;
}

// Evolve running alphas
// eq. (31) from (ii)
Double_t GEW::RunningAlphaQCD::AlphaStrong(Int_t nf, Double_t mu, Double_t lambda) const
{

   // protection against unphysical values of lambda
   if (TMath::Abs(lambda)<10e-6 || TMath::Abs(lambda)>10e6){
     return 0.;
   }
   Double_t L    = 2.0*( TMath::Log( TMath::Abs( mu/lambda ) ) );
   Double_t logL = TMath::Log( TMath::Abs( L ) );
   Double_t alphas = ( 1/(m_b0[nf]*L)*
                       ( 1.0 - m_b1[nf]*logL/(m_b0[nf]*m_b0[nf]*L) + 1/(m_b0[nf]*m_b0[nf]*L*L)*
                         ( GMath::IPow(m_b1[nf]/m_b0[nf],2)
                           *( GMath::IPow(logL,2) - logL -1.0 ) + m_b2[nf]/m_b0[nf] )
                         + 1/GMath::IPow(m_b0[nf]*L,3)*
                         ( GMath::IPow(m_b1[nf]/m_b0[nf],3)
                           *( - GMath::IPow(logL,3) +5/2.0*logL*logL + 2.0*logL - 0.5 )
                           - 3.0*m_b1[nf]*m_b2[nf]*logL/(m_b0[nf]*m_b0[nf]) 
                           + m_b3[nf]/(2.0*m_b0[nf]) ) ) );
   return alphas;	   
}

// matching condition for decreasing nf
// inverted eqs. (4) and (6) from (i)
// nf belongs to the old alphas
Double_t GEW::RunningAlphaQCD::AlphasMatchDown(Int_t nf, Double_t alphas, Double_t mu_nf, Double_t qmass) const
{
   Double_t x  = 2.0*TMath::Log( mu_nf/qmass );
   
   Double_t C1 = x/6.0;
   Double_t C2 = m_c20 + 19/24.0*x + x*x/36.0;
   Double_t C3 = ( m_c30[nf] + (241/54.0 + 13/4.0*m_c20 - (325/1728.0 + m_c20/6.0)*nf)*x
                   + 511/576.0*x*x + x*x*x/216.0 );

   Double_t A1 = -C1;
   Double_t A2 = 2*C1*C1 - C2;
   Double_t A3 = 3.0*C1*C1*C1 + C2*C1 - C3;
   Double_t As = alphas/TMath::Pi();
   
   Double_t alphasnew = alphas*(1.0 + A1*As + A2*As*As + A3*As*As*As);

   return alphasnew;
}

// matching condition for increasing nf
// eqs. (4) and (6) from (i)
// nf belongs to the new alphas
Double_t GEW::RunningAlphaQCD::AlphasMatchUp(Int_t nf, Double_t alphas, Double_t mu_nf, Double_t qmass) const
{
   Double_t x  = 2.0*TMath::Log( mu_nf/qmass );
   
   Double_t C1 = x/6.0;
   Double_t C2 = m_c20 + 19/24.0*x + x*x/36.0;
   Double_t C3 = ( m_c30[nf] + (241/54.0 + 13/4.0*m_c20 - (325/1728.0 + m_c20/6.0)*nf)*x
                   + 511/576.0*x*x +x*x*x/216.0 );

   Double_t As = alphas/TMath::Pi();

   Double_t alphasnew = alphas*(1 + C1*As + C2*As*As + C3*As*As*As);

   return alphasnew;
}

// computes new Lambda for different nf  
// for reference read paper (ii)
Double_t GEW::RunningAlphaQCD::LambdaAtThreshold( Int_t nf, Double_t alphas, Double_t mu) const
{
   Double_t pi   = TMath::Pi(); 
   Double_t b0   = m_b0[nf]*pi;
   Double_t b1   = m_b1[nf]*pi*pi/b0;
   Double_t b2   = m_b2[nf]*pi*pi*pi/b0;
   Double_t b3   = m_b3[nf]*pi*pi*pi*pi/b0;
   Double_t C    = b1/b0*TMath::Log(b0);

   Double_t asPi = alphas/pi;

   Double_t T = 1/b0*( 1.0/asPi + b1*TMath::Log(asPi) + (b2 - b1*b1)*asPi
                       + (b3/2.0 - b1*b2 + b1*b1*b1/2.0)*asPi*asPi ) + C;

   return mu*TMath::Sqrt(TMath::Exp(-T));
}

// thisPoiter = NULL
GEW::RunningAlphaQCD* GEW::RunningAlphaQCD::m_thisPointer = NULL;

// static functions for Runge-Kutta method
Double_t GEW::RunningAlphaQCD::GetRGE1( Double_t x, Double_t y )
{
   return GetThisPointer()->RGE( 1, x, y );
}  

Double_t GEW::RunningAlphaQCD::GetRGE2( Double_t x, Double_t y )
{
   return GetThisPointer()->RGE( 2, x, y );
}  

Double_t GEW::RunningAlphaQCD::GetRGE3( Double_t x, Double_t y )
{
   return GetThisPointer()->RGE( 3, x, y );
}  

Double_t GEW::RunningAlphaQCD::GetRGE4( Double_t x, Double_t y )
{
   return GetThisPointer()->RGE( 4, x, y );
}  

Double_t GEW::RunningAlphaQCD::GetRGE5( Double_t x, Double_t y )
{
   return GetThisPointer()->RGE( 5, x, y );
}  

Double_t GEW::RunningAlphaQCD::GetRGE6( Double_t x, Double_t y )
{
   return GetThisPointer()->RGE( 6, x, y );
}  

// RGE
// for instance: eq. (A.1) from (iii)
Double_t GEW::RunningAlphaQCD::RGE( Int_t nf, Double_t /* x */, Double_t y )
{
   return -y*y*( m_b0[nf] + m_b1[nf]*y + m_b2[nf]*y*y + m_b3[nf]*y*y*y );
}

Double_t GEW::RunningAlphaQCD::GetLambdaMS5()
{
   // for kFitLambda : return parameter from data card
   if      ( GetQCDType() == kFitLambda ) return p_InputValueLambdaMS5;

   // for kFitAlphas : numerical calculation of lambdaMS5
   else if ( GetQCDType() == kFitAlphas ){
      GRootFinder rootFinder( &RootAlphas, 0.05, 0.5 );
      return rootFinder.Root( p_InputValueAlphasMZ );
   }
   else{
      m_logger << kFATAL << "unknown value for \"p_QCDType\": \"" << GetQCDType() << "\""
               << GEndl;
      return 0;
   }
}

// static function for root finding
Double_t GEW::RunningAlphaQCD::RootAlphas( Double_t lambda )
{
   return GetThisPointer()->AlphasAtMZ( lambda );
}

// Get EvolveAlphas and initialise non static member values
// particulary p_MZ
Double_t GEW::RunningAlphaQCD::AlphasAtMZ( Double_t lambda )
{
   return AlphaStrong( 5, p_MZ, lambda );
}
