/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : WParametrisation                                                      *
 *                                                                                *
 * Description:                                                                   *
 *      W Boson decay / total width                                               *
 *                                                                                *
 * Sources:                                                                       *
 *      (i)  G.-C. Cho, K. Hagiwara, Y. Matsumoto and D. Nomura                   *
 *           JHEP 1111 (2011) 068, (arXiv: 1104.1769v1)                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GConstants.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GEW/WParametrisation.h"

using namespace Gfitter;

ClassImp(GEW::WParametrisation)

GEW::WParametrisation::WParametrisation()
   : WBase(), 
     m_isUpToDate_Update( kFALSE )     
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
   
   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }

   BookTheory   ( "GEW::MH"                    , &t_MH );
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ"        , &t_AlphasMZ );  

   BookTheory   ( "GEW::MW"                    , & m_MW );
}

void GEW::WParametrisation::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

void GEW::WParametrisation::Update()
{
   if (m_isUpToDate_Update) return;

   // now, it is uptodate (I mean... it will be)
   m_isUpToDate_Update = kTRUE;

   Double_t MW     = GetMW();
   Double_t MH     = GetMH().GetValue(); 
   if (m_logMH) MH = TMath::Exp( GetMH().GetValue() );
   
   // eq. 22 and 23e from (i) 
   Double_t RW = 2.1940 - 0.16*(TMath::Log(1+std::pow(23./MH,2)) - TMath::Log(1+std::pow(23./100,2))); 

   // eq 24 from (i)
   Double_t xs = (GetAlphasMZ() - 0.118)/0.003;

   // eq. 20 from (i)
   m_gammaW = 0.33904*MW*MW*MW*p_GF*(1. + 8.478e-3*RW + 0.00065*xs);
  
   // now, parameters are up-to-date
   SetUpToDate();
}

