/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Z0Parametrisation                                                     *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory of the parametrisation option                            *
 *      Computes effective weak mixing angle and Partial Z widths                 *
 *                                                                                *
 * Sources:                                                                       *
 *      (i)   G.-C. Cho, K. Hagiwara, Y. Matsumoto and D. Nomura                  *
 *            JHEP 1111 (2011) 068, [arXiv:1104.1769 [hep-ph]]                    *
 *      (ii)  K. Hagiwara, S. Matsumoto, D. Haidt and C. S. Kim                   *
 *            Z. Phys. C 64 (1994) 559, [hep-ph/9409380]                          *
 *      (iii) D. Bardin, G. Passarino, The Standard Model in the Making,          *
 *            Oxford (1999), Page 496- , and references therein                   *
 *      (iv)  P.A. Baikov, K.G. Chetyrkin, J.H. Kuhn, J. Rittinger                *
 *            (arXiv:1201.5804)                                                   *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GConstants.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GEW/Z0Parametrisation.h"
#include "GEW/Sin2ThetaF.h"
#include "GEW/RadiatorFunctions.h"
#include "GEW/MH.h"

using namespace std;
using namespace Gfitter;
using namespace Gfitter::GTypes;

ClassImp(GEW::Z0Parametrisation)

GEW::Z0Parametrisation::Z0Parametrisation()
   : Z0Base(),
     m_isUpToDate_Update( kFALSE )
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
 
   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }
   
   BookTheory( "GEW::RadiatorFunctions"     , &t_radFun );
}

void GEW::Z0Parametrisation::Initialise()
{
   // table 4 from (i) 
   // non-factorizable EW*QCD corrections
   m_EWQCD[0] = 0;           // neutrino
   m_EWQCD[1] = 0;           // electron, muon, tau
   m_EWQCD[2] = -0.113e-3;   // up
   m_EWQCD[3] = -0.160e-3;   // down
   m_EWQCD[4] =  m_EWQCD[2]; // charm
   m_EWQCD[5] =  m_EWQCD[3]; // strange
   m_EWQCD[6] = -0.040e-3;   // bottom
 
   // table 4 from (i)
   // imaginary part of Z/gamma interfernce term
   m_ImKappa[0] = 0;         // neutrino
   m_ImKappa[1] = 0.0000368; // electron, muon, tau
   m_ImKappa[2] = 0.0000146; // up
   m_ImKappa[3] = 0.0000032; // down
   m_ImKappa[4] = 0.0000146; // charm
   m_ImKappa[5] = 0.0000032; // strange
   m_ImKappa[6] = 0.0000026; // bottom
}

void GEW::Z0Parametrisation::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

void GEW::Z0Parametrisation::Update()
{
   if (m_isUpToDate_Update) return;

   // now, it is uptodate (I mean... it will be)
   m_isUpToDate_Update = kTRUE;

   Double_t MH        = GetMH().GetValue(); 
   if (m_logMH) MH = TMath::Exp( GetMH().GetValue() );

   // eq. 24 from (i)
   Double_t xt = (p_mt - 172.)/3.;
   Double_t xh = TMath::Log( MH/100. )/TMath::Log( 10. );
   Double_t xs = (GetAlphasMZ() - 0.118)/0.003;
   // eq. 15 from (i)
   Double_t xalpha = (GetDAlphaQED().DAlphaHad5() - 0.0277)/0.0003;

   // eq. 23a and 23b from (i)
   m_Delta_SZ = 0.2217*xh - 0.1188*xh*xh + 0.0320*xh*xh*xh - 0.0014*xt + 0.0005*xs;
   m_Delta_TZ = (-0.0995*xh - 0.2858*xh*xh + 0.1175*xh*xh*xh + 0.0367*xt + 0.00026*xt*xt 
                 -0.0017*xh*xt - 0.0033*xs - 0.0001*xt*xs );

   // eq. 13a and 13b from (i)
   Double_t Delta_gbarZ2 = 0.00412*m_Delta_TZ;
   Double_t Delta_sbar2  = 0.00360*m_Delta_SZ - 0.00241*m_Delta_TZ + 0.00011*xalpha;

   // eq. 12a-12i from (i)
   m_gLv =  0.50199 + 0.45250*Delta_gbarZ2 + 0.00469*Delta_sbar2;
   m_gRv =  0;
   m_gLe = -0.26920 - 0.24338*Delta_gbarZ2 + 1.00413*Delta_sbar2;
   m_gRe =  0.23207 + 0.20912*Delta_gbarZ2 + 1.00784*Delta_sbar2;
   m_gLu =  0.34675 + 0.31309*Delta_gbarZ2 - 0.66793*Delta_sbar2;
   m_gRu = -0.15470 - 0.13942*Delta_gbarZ2 - 0.67184*Delta_sbar2;
   m_gLd = -0.42434 - 0.38279*Delta_gbarZ2 + 0.33166*Delta_sbar2;
   m_gRd =  0.07734 + 0.06971*Delta_gbarZ2 + 0.33590*Delta_sbar2;
   // plus eq. 25a, 25b from (i)
   m_gLb = -0.42116 - 0.38279*Delta_gbarZ2 + 0.33166*Delta_sbar2 - 0.000058*xh + 0.000128*xt;
   m_gRb =  0.07742 + 0.06971*Delta_gbarZ2 + 0.33590*Delta_sbar2 - 0.000042*xh - 0.000025*xh*xh*xh*xh; 
   
   // vector radiator functions (not used)
   // table 4 from (i) 
   m_CV[0] = 1;                  // neutrino
   m_CV[1] = 1;                  // electron, muon, tau
   m_CV[2] = 3.1666 + 0.0030*xs; // up
   m_CV[3] = 3.1667 + 0.0030*xs; // down
   m_CV[4] = 3.1667 + 0.0030*xs; // charm
   m_CV[5] = 3.1667 + 0.0030*xs; // strange
   m_CV[6] = 3.1185 + 0.0030*xs; // bottom

   // axial radiator functions (not used)
   // table 4 from (i) 
   m_CA[0] = 1;                               // neutrino
   m_CA[1] = 1;                               // electron, muon, tau
   m_CA[2] = 3.1377 + 0.00014*xt + 0.0041*xs; // up
   m_CA[3] = 3.0956 - 0.00015*xt + 0.0019*xs; // down
   m_CA[4] = 3.1369 + 0.00014*xt + 0.0043*xs; // charm
   m_CA[5] = 3.0956 - 0.00015*xt + 0.0019*xs; // strange
   m_CA[6] = 3.0758 - 0.00015*xt + 0.0028*xs; // bottom

   SetUpToDate();
}

// effective weak mixing angle
// see also class Sin2ThetaF.h
Double_t GEW::Z0Parametrisation::GetSin2Eff( GTypes::Particle ParticleType )
{   
   Double_t sineff = 0;

   sineff = GetSinEffF().GetSin2ThetaF( ParticleType );
   
   return sineff;
}

// relation between left-right and vector coupling
Double_t GEW::Z0Parametrisation::gVf( GTypes::Particle ParticleType, Double_t /*T3*/ )
{
   Update();
   Double_t coupl = 0;

   switch(ParticleType){
   case kElectron:
   case kMuon:
   case kTau:
      coupl = m_gLe + m_gRe;
      break;
   case kNeutrino:
      coupl = m_gLv + m_gRv;
      break;
   case kUp:
   case kCharm:
      coupl = m_gLu + m_gRu;
      break;
   case kDown:
   case kStrange:
      coupl = m_gLd + m_gRd;
      break;
   case kBottom:
      coupl = m_gLb + m_gRb;
      break;
   default:
      m_logger << kFATAL << "False input for ParticleType" << GTypes::GetName(ParticleType) << GEndl;
      break;
   }

   return coupl;
}

// relation between left-right and axial coupling
Double_t GEW::Z0Parametrisation::gAf( GTypes::Particle ParticleType, Double_t /*T3*/ )
{
   Update();
   Double_t coupl = 0;
 
   switch(ParticleType){
   case kElectron:
   case kMuon:
   case kTau:
      coupl  = m_gLe - m_gRe;
      break;
   case kNeutrino:
      coupl  = m_gLv - m_gRv;
      break;
   case kUp:
   case kCharm:
      coupl  = m_gLu - m_gRu;
      break;
   case kDown:
   case kStrange:
      coupl  = m_gLd - m_gRd;
      break;
   case kBottom:
      coupl  = m_gLb - m_gRb;
      break;
   default:
      m_logger << kFATAL << "False input for ParticleType" << GTypes::GetName(ParticleType) << GEndl;
      break;
   }
   
   //
   return coupl;
}

// partial width of Z boson
// eq. 26 from (i) but using different radiator functions includig more radiative corrections
// (more details can be found in the RaditorFunction class)
Double_t GEW::Z0Parametrisation::GammaZff( GTypes::Particle ParticleType, Double_t Charge, Double_t mf )
{
   Update();
   
   Double_t Gamma0 = p_GF*GMath::IPow(p_MZ,3) / (6.0*GMath::Sqrt2()*TMath::Pi());
   Double_t aQEDMZ = GConstants::alphaQED()/( 1 - GetDAlphaQED().DAlphaQEDMZt() );
   Double_t T3     = 0.5*TMath::Sign(1.0,Charge+0.1);
   //Double_t gV     = gVf( ParticleType, T3 );
   Double_t gA     = gAf( ParticleType, T3 );
   Double_t RadV   = 1;
   Double_t RadA   = 1;
   Double_t beta   = 1;
   Double_t ratio  = 1;
   int      index  = 0;

   switch(ParticleType){
   case kNeutrino :
      index  = 0;
      RadV   = 1;
      RadA   = 1;
      break;
   case kElectron :
   case kMuon     :
   case kTau      :
      index  = 1;
      // eq. 3.13a, 3.13b, and 3.14 from (ii)
      ratio  = std::pow( mf/p_MZ, 2 );
      beta   = std::sqrt( 1 - 4*ratio );
      RadV   = beta*(3-beta*beta)/2.*(1.0 + 0.75*aQEDMZ/TMath::Pi()*Charge*Charge);
      RadA   = beta*beta*beta*(1.0 + 0.75*aQEDMZ/TMath::Pi()*Charge*Charge);
      break;
   case kUp       : 
      index  = 2;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kDown     :
      index  = 3;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kCharm    :
      index  = 4;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kStrange  :
      index  = 5;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   case kBottom   :
      index  = 6;
      RadV   = 3*GetRadFun().GetRVq( ParticleType, Charge);
      RadA   = 3*GetRadFun().GetRAq( ParticleType, Charge, T3);
      break;
   default:
      m_logger << kFATAL << "false input for ParticleType : " << GTypes::GetName(ParticleType) << GEndl;
      break;
   } 

   // instead using gV and gA, take ratio of couplings, since sineff more accurate
   Double_t sineff     = GetSin2Eff(ParticleType);
   Double_t RatioCplgs = 1.0 - 4.0*TMath::Abs(Charge)*sineff;
   Double_t Gamma = Gamma0*( gA*gA*( RatioCplgs*RatioCplgs*RadV + RadA ) + m_ImKappa[index]*RadV ) + m_EWQCD[index];

   return Gamma;
}

// please note: Gamma_had != Gamma_u + Gamma_d + Gamma_c + Gamma_s + Gamma_b (see for instance ref. (iii) Bardin+Passarino on page 500)
// the singlet vector contribution cannot be assigned to single quark flavor (take it only into account for full hadronic width) 
Double_t GEW::Z0Parametrisation::GetGaZhad()
{
   Update();
   
   Double_t ghad   =  GetGaZup() + GetGaZdown() + GetGaZcharm() + GetGaZstrange() + GetGaZbottom();
   
   GTypes::Particle types[] = { GTypes::kUp, GTypes::kDown, GTypes::kCharm, GTypes::kStrange, GTypes::kBottom };
   Double_t charges[] = { GMath::TwoThird(), -GMath::OneThird(), GMath::TwoThird(), -GMath::OneThird(), -GMath::OneThird() };
   
   // ref. (iii)  Bardin+Passarino, eq. (12.42)
   // and ref. (iv) Baikov et al, eq. (2) and eq. (3)
   Double_t sum = 0;
   for( int i = 0; i < 5; i++ ){
      Double_t T3 = 0.5*TMath::Sign(1.0,charges[i]+0.1);
      Double_t gV = gAf( types[i], T3 ) - 2.0*charges[i]*GetSin2Eff( types[i] ); // gVf( types[i], T3 );
      sum += gV; // note: vf = 2*gVf
   }
 
   Double_t Gamma0 = p_GF*GMath::IPow(p_MZ,3) / (6.0*GMath::Sqrt2()*TMath::Pi());
   ghad += Gamma0*3*sum*sum*GetRadFun().GetrVS();
   
   return ghad;
}
