/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Z0Parametrisation                                                     *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory of the parametrisation option                            *
 *      Computes effective weak mixing angle and Partial Z widths                 *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef GEW_Z0Parametrisation
#define GEW_Z0Parametrisation

#include "Gfitter/GConstants.h"

#include "GEW/Z0Base.h"

using namespace Gfitter;





namespace GEW {

   class Sin2ThetaF;
   class RadiatorFunctions;
   class DAlphaQED;
   class MH;
   
   class Z0Parametrisation : public Z0Base {

   public:
      
      Z0Parametrisation();
      virtual ~Z0Parametrisation() {}
      
      void Initialise();

      // accessors
      virtual Double_t GetSin2Eff(Gfitter::GTypes::Particle p);
      virtual Double_t GetGaZelectron();
      virtual Double_t GetGaZmuon();
      virtual Double_t GetGaZtau();
      virtual Double_t GetGaZneutrino();
      virtual Double_t GetGaZup();
      virtual Double_t GetGaZdown();
      virtual Double_t GetGaZstrange();
      virtual Double_t GetGaZcharm();
      virtual Double_t GetGaZbottom();
      virtual Double_t GetGaZhad();
      virtual Double_t GetDeltaS();
      virtual Double_t GetDeltaT();

      // effective vector and axial couplings
      virtual Double_t GetgAeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t ch );
      virtual Double_t GetgVeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t ch );

   protected:
      
      // partyial Z weidth
      Double_t GammaZff( Gfitter::GTypes::Particle particle, Double_t ch, Double_t mf =0  );  

      // update of flags in case of parameter change
      void UpdateLocalFlags( Gfitter::GReference& ref );
   
      // update parameters
      void Update();

      Bool_t m_logMH;
  
   private:

      // axial and vector coupling
      Double_t gVf( GTypes::Particle ParticleType, Double_t T3 );
      Double_t gAf( GTypes::Particle ParticleType, Double_t T3 );
      
      // boolean flags to track recomputation
      Bool_t m_isUpToDate_Update; 

      Double_t m_Delta_SZ;
      Double_t m_Delta_TZ;

      // left/right couplings
      double m_gLv;
      double m_gRv;
      double m_gLe;
      double m_gRe;
      double m_gLu;
      double m_gRu;
      double m_gLd;
      double m_gRd;
      double m_gLb;
      double m_gRb;
   
      // non-factrorizable EW*QCD corrctions to Z->ff
      Double_t m_EWQCD[7];
        
      // imaginary part of Z/gamma interfernce term
      Double_t m_ImKappa[7];

      // axial and vector radiator functions
      Double_t m_CV[7];
      Double_t m_CA[7];
             
      // reference to radiator function
      RadiatorFunctions& GetRadFun() const { return (RadiatorFunctions&)*t_radFun; }
      Gfitter::GTheoryRef    t_radFun;
      
      ClassDef(Z0Parametrisation,0)
   };
}

//--------------------------- inline function for fast access ------------------------

inline Double_t GEW::Z0Parametrisation::GetGaZelectron()
{
   return GammaZff( GTypes::kElectron, -1, GConstants::me() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZmuon()
{
   return GammaZff( GTypes::kMuon, -1, GConstants::mmu() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZtau()
{   
   return GammaZff( GTypes::kTau, -1, GConstants::mtau() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZneutrino()
{   
   return GammaZff( GTypes::kNeutrino, 0, 0);
}

inline Double_t GEW::Z0Parametrisation::GetGaZup()
{   
   return GammaZff( GTypes::kUp, GMath::TwoThird() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZdown()
{   
   return GammaZff( GTypes::kDown, -GMath::OneThird() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZstrange()
{   
   return GammaZff( GTypes::kStrange, -GMath::OneThird() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZcharm()
{   
   return GammaZff( GTypes::kCharm, GMath::TwoThird() );
}

inline Double_t GEW::Z0Parametrisation::GetGaZbottom()
{   
   return GammaZff( GTypes::kBottom, -GMath::OneThird() );
}

// effective couplings / A = axial, V = vector

inline Double_t GEW::Z0Parametrisation::GetgAeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t /*ch*/ )
{   
   Double_t gA = gAf( particle, T3 );
  
   return gA;
}    

inline Double_t GEW::Z0Parametrisation::GetgVeff( Gfitter::GTypes::Particle particle, Double_t T3, Double_t /*ch*/ )
{
   Double_t gV = gVf( particle, T3 );
         
   return gV;
}

inline Double_t GEW::Z0Parametrisation::GetDeltaS()
{   
   Update();
   return m_Delta_SZ;
}

inline Double_t GEW::Z0Parametrisation::GetDeltaT()
{   
   Update();
   return m_Delta_TZ;
}

#endif
