/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : InertDoubletModel                                                     *
 *                                                                                *
 * Description:                                                                   *
 *      implementation                                                            *
 *                                                                                *
 * Source:                                                                        *
 *     - R. Barbieri, L. J. Hall and V. S. Rychkov                                *
 *       Phys. Rev.  D 74, 015007 (2006), [arXiv:hep-ph/0603188]                  *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"
#include "TF1.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"

#include "GSTU/InertDoubletModel.h"

using namespace Gfitter;
using namespace std;

const Bool_t UseAnalyticalSolutionForS = kTRUE;

ClassImp(GSTU::InertDoubletModel)

GSTU::InertDoubletModel::InertDoubletModel()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // book InertDoubletModel parameters
   BookParameter( "MHcharge",  &p_MHcharge ); 
   BookParameter( "ML",        &p_ML ); 
   BookParameter( "DeltaM",    &p_DeltaM ); 
}


void GSTU::InertDoubletModel::Initialise(){}

// eq. (65) from hep-ph/0603188
Double_t GSTU::InertDoubletModel::GetS_Numerical()
{
   TF1 *f = new TF1( "Sfunc", Sfunc, 0, 1, 3 );
   f->SetParameter( 0, p_ML ); 
   f->SetParameter( 1, p_ML + p_DeltaM ); 
   f->SetParameter( 2, p_MHcharge ); 
  
   return f->Integral(0,1);   
}

// eq. (65) from hep-ph/0603188 analytically integrated with Mathematica
Double_t GSTU::InertDoubletModel::GetS_Analytical()
{
   // choose H0 to be light Higgs
   Double_t MH02  = TMath::Power( p_ML, 2 );
   Double_t MA02  = TMath::Power( p_ML + p_DeltaM, 2 );
   Double_t MHch2 = TMath::Power( p_MHcharge, 2 );
   Double_t DM    = MA02 - MH02;

   Double_t S = 0.5/TMath::Pi();
   S *= ( TMath::Log(MH02/MHch2)/6.0 - 5.0/36.0 + MH02*MA02/3.0/TMath::Power( DM, 2 )
          + MA02*MA02*(MA02 - 3.0*MH02)/6.0/TMath::Power( DM, 3 )*TMath::Log(MA02/MH02) );
  
   return S;
}

Double_t GSTU::InertDoubletModel::GetS()
{
   Double_t S( 0 );

   if (UseAnalyticalSolutionForS) S = GetS_Analytical();
   else                           S = GetS_Numerical();

   return S;
}

// eq. (65) from hep-ph/0603188
Double_t GSTU::InertDoubletModel::Sfunc( Double_t *x, Double_t *par )
{
   double xx = x[0];
   double ms = par[0];
   double ma = par[1];
   double mh = par[2];
   
   if( xx == 0 || xx == 1 ) return 0;
   
   return 1/(2*TMath::Pi())*xx*(1-xx)*TMath::Log( (xx*ms*ms + (1-xx)*ma*ma)/(mh*mh) );
}

// eq. (27) from hep-ph/0603188
Double_t GSTU::InertDoubletModel::GetT()
{
   Double_t vev   = GConstants::vev();
   Double_t alpha = GConstants::alphaQED();
   Double_t pi    = TMath::Pi();
 
   return 1/(32*pi*pi*alpha*vev*vev)*( F(p_MHcharge,p_ML) + F(p_MHcharge,p_ML+p_DeltaM) - F(p_ML,p_ML+p_DeltaM) );
}

// eq. (28) from hep-ph/0603188
Double_t GSTU::InertDoubletModel::F( Double_t m1, Double_t m2 )
{
   if( m1 == m2 ) return 0;

   Double_t m12 = m1*m1;
   Double_t m22 = m2*m2;
   
   return (m12+m22)/2. - m12*m22/(m12-m22)*TMath::Log(m12/m22);
}
