/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : LED_amu                                                               *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * Source:                                                                        *
 *      - R. Contino, L. Pilo, R. Rattazzi and A. Strumia                         *
 *        JHEP 0106, 005 (2001), [arXiv:hep-ph/0103104]                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include <iostream>
#include <cmath>
#include "Gfitter/GConstants.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"

#include "GSTU/LED_amu.h"

#include "TMath.h"

using namespace Gfitter;

ClassImp(GSTU::LED_amu)


GSTU::LED_amu::LED_amu(): Gfitter::GTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   Int_t value = gStore()->GetVariable( "GSTUFlags::LED_Ndim" )->GetIntValue();
   m_logger << kINFO << "Using: \"" << value << "\" dimensions for large extra dimensions!" << GEndl;
   m_ndim = value;

   bool value2 = gStore()->GetVariable( "GSTUFlags::LED_LogScale" )->GetBoolValue();
   m_logger << kINFO << "Using: \"" << value2 << "\" logarithmic scale!" << GEndl;
   m_logScale = value2;
   
   if( m_logScale ){
      BookParameter( "logMD",      &p_logMD );
      BookParameter( "logLambdaMD", &p_logLambdaMD );
   }
   else{
      BookParameter( "MD",       &p_MD );
      BookParameter( "LambdaMD", &p_LambdaMD );
   }
   
}

// for references see hep-ph/0103104v2 page 25-26
Double_t GSTU::LED_amu::GetTheoryPrediction()
{
   Double_t mu = 0.001*GConstants::mmu(); // use TeV
   Double_t MD, lambdaMD;
   if( m_logScale ){
      MD       = std::pow( 10, p_logMD );
      lambdaMD = std::pow( 10, p_logLambdaMD );
   }   
   else{
      MD       = p_MD;
      lambdaMD = p_LambdaMD;
   }
   
   double da_mu = mu*mu/(MD*MD)*std::pow(lambdaMD, m_ndim)*(34.0 + 11.0*m_ndim)
      /( 96.*TMath::Gamma(2. + m_ndim/2.)*pow(TMath::Pi(), 2.-m_ndim/2.) );
   
   return da_mu;

}
