/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : TheoryHandler                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"
#include <string>

#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"

#include "GSTU/TheoryHandler.h"
#include "GSTU/LittlestHiggs.h"
#include "GSTU/RandallSundrum.h"
#include "GSTU/TechniColor.h"
#include "GSTU/ExtraFamilies.h"
#include "GSTU/UED.h"

using namespace Gfitter;

ClassImp(GSTU::TheoryHandler)

GSTU::TheoryHandler::TheoryHandler()
   : Gfitter::GTheory()
   , m_byPassNaN(false)
   , AddSM( kFALSE )
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   if ( gStore()->ExistVariable( "GSTUFlags::ByPassNaN" ) ) {
     m_byPassNaN = gStore()->GetVariable( "GSTUFlags::ByPassNaN" )->GetBoolValue();
   }

   const TString& value = gStore()->GetVariable( "GSTUFlags::Theory" )->GetStringValue();
   m_logger << kINFO << "Using Theory: \"" << value << "\"" << GEndl;
   TString replace = value;

   if( replace.EndsWith("+SM") ){
      AddSM = kTRUE;
      Int_t pos = replace.Last('+');
      replace.Remove(pos,3);
   }

   if     ( replace == "STU" )               m_replace = kSTU;
   else if( replace == "LittlestHiggs" )     m_replace = kLittlestHiggs;
   else if( replace == "Randall-Sundrum" )   m_replace = kRandallSundrum;
   else if( replace == "Randall-Sundrum_CS" ) m_replace = kRandallSundrum_CS;
   else if( replace == "TechniColor" )       m_replace = kTechniColor;
   else if( replace == "ExtraFamilies" )     m_replace = kExtraFamilies;
   else if( replace == "SUSY")               m_replace = kSUSY;
   else if( replace == "UED" )               m_replace = kUED;
   else if( replace == "TwoHDM" )            m_replace = kTwoHDM;
   else if( replace == "InertDoubletModel" ) m_replace = kIDM;
   else {
      m_logger << kFATAL << "unknown theory for \"GSTUFlags::Theory\": \"" << replace << "\""
               << ". Possible are: \"STU\", \"LittlestHiggs\", \"Randall-Sundrum\",  \"InertDoubletModel\", \"Randall-Sundrum_CS\", \"TechniColor\", \"SUSY\", \"TwoHDM\", \"InertDoubletModel\" and \"UED\""
               << GEndl;
   }
   
   if ( AddSM )
      BookTheory( "GOblique::EpsilonParametrisation" ,&t_SM );

   // STU Parameters
   if( IsSTU() ){
      BookParameter( "S", &p_S );
      BookParameter( "T", &p_T );
      BookParameter( "U", &p_U );
      BookParameter( "V", &p_V );
      BookParameter( "W", &p_W );
      BookParameter( "X", &p_X );
      BookParameter( "dgRb", &p_dgRb );
      BookParameter( "dgLb", &p_dgLb );
   }
   // Littlest Higgs with T-Parity
   else if( IsLittlestHiggs() ){
      BookTheory   ( "GSTU::LittlestHiggs", &t_theory );
   }
   // Randall-Sundrum Model
   else if( IsRandallSundrum() ){
      BookTheory   ( "GSTU::RandallSundrum", &t_theory );
   }
   // Randall-Sundrum Model with Custodial Symmetry
   else if( IsRandallSundrum_CS() ){
      BookTheory   ( "GSTU::RandallSundrum_CustSym", &t_theory );
   }
   // TechniColor Model
   else if( IsTechniColor() ){
      BookTheory   ( "GSTU::TechniColor", &t_theory );
   }
   // 4th generation
   else if( IsExtraFamilies() ){
      BookTheory   ( "GSTU::ExtraFamilies", &t_theory );
   }
   // one universal extra dimension
   else if( IsUED() ){
      BookTheory   ( "GSTU::UED", &t_theory );
   }
   // two Higgs doublet model
   else if( IsTwoHDM() ){
      BookTheory   ( "GSTU::TwoHDM", &t_theory );
   }
   // inert Higgs doublet model
   else if( IsInertDoublet() ){
      BookTheory   ( "GSTU::InertDoubletModel", &t_theory );
   }
   // SUSY --> MB: I've put the corresponding files GSTUSusy.* in the gsusy/ directory
   // to avoid unnecessary dependencies of gstu on gsusy.
   else if( IsSUSY() ) {
      BookTheory   ( "GSUSY::GSTUSusy", &t_theory );
   }  
   else {
      m_logger << kFATAL << "<TheoryHandler>: Unknown theory!!!" << GEndl;
   }
}

// S parameter
Double_t GSTU::TheoryHandler::GetS()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_S;
   }  
   else {
      retval = GetTheory().GetS();
   }
 
   if( AddSM )
      retval += GetSM().GetS();

   return checkNaN(retval, 0.1);
}

// T parameter
Double_t GSTU::TheoryHandler::GetT()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_T;
   }
   else {
      retval = GetTheory().GetT();
   }

   if( AddSM )
      retval += GetSM().GetT();
   
   return checkNaN(retval, 0.1);
}

// U parameter
Double_t GSTU::TheoryHandler::GetU()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_U;
   }
   else {
      retval = GetTheory().GetU();
   }
   
   if( AddSM )
      retval += GetSM().GetU();

   return checkNaN(retval, 0.1);
}

// V parameter
Double_t GSTU::TheoryHandler::GetV()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_V;
   }
   else {
      retval = GetTheory().GetV();
   }

   return checkNaN(retval);
}

// W parameter
Double_t GSTU::TheoryHandler::GetW()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_W;
   }
   else {
      retval = GetTheory().GetW();
   }

   return checkNaN(retval);
}

// X parameter
Double_t GSTU::TheoryHandler::GetX()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_X;
   }
   else {
      retval = GetTheory().GetX();
   }

   return checkNaN(retval);
}


// additional correction to Z->bb couplings
// right
Double_t GSTU::TheoryHandler::Get_deltagRb()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_dgRb;
   }
   else {
      retval = GetTheory().Get_deltagRb();
   }

   return checkNaN(retval);
}

// left
Double_t GSTU::TheoryHandler::Get_deltagLb()
{
   Double_t retval = 0;
   
   if( IsSTU() ){
      retval = p_dgLb;
   }
   else {
      retval = GetTheory().Get_deltagLb();
   }

   return checkNaN(retval);
}


// MB: nan crash protection
Double_t 
GSTU::TheoryHandler::checkNaN(const Double_t& value, const Double_t& defaultVal) const
{
   if (!m_byPassNaN) return value; 
   if ( TMath::IsNaN( value ) ){
      m_logger << kWARNING << "NaN detected. By-pass, returning dummy value : " << defaultVal << GEndl;
      return defaultVal;
   } else return value;
}

