/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : TheoryHandler                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      class replaces STU corrections by the chosen model                        *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
**********************************************************************************/

#ifndef GSTU_TheoryHandler
#define GSTU_TheoryHandler

#include "Gfitter/GTheory.h"

#include "GOblique/EpsilonParametrisation.h"




namespace GSTU {

   class TheoryBase;
   
   class TheoryHandler : public Gfitter::GTheory {

   public:
      
      TheoryHandler();
      ~TheoryHandler() {}

      // STU corrections
      Double_t GetS();
      Double_t GetT();
      Double_t GetU();
      Double_t GetV();
      Double_t GetW();
      Double_t GetX();
      // additional corrections to Z->bb couplings
      // in STU case corrections are set to Zero
      Double_t Get_deltagRb();
      Double_t Get_deltagLb();

   protected:

      // STU Parameters
      Gfitter::GParameterRef p_S;
      Gfitter::GParameterRef p_T;
      Gfitter::GParameterRef p_U;
      Gfitter::GParameterRef p_V;
      Gfitter::GParameterRef p_W;
      Gfitter::GParameterRef p_X;
      Gfitter::GParameterRef p_dgRb;
      Gfitter::GParameterRef p_dgLb;

      // reference to TheoryBase
      TheoryBase& GetTheory() const { return (TheoryBase&)*t_theory; }
      Gfitter::GTheoryRef t_theory;

      // reference SM calculation
      GOblique::EpsilonParametrisation& GetSM() const { return (GOblique::EpsilonParametrisation&)*t_SM; }      
      Gfitter::GTheoryRef t_SM; 

      // MB: nan protection
      Double_t checkNaN(const Double_t& value, const Double_t& defaultVal = 0) const;

      Bool_t m_byPassNaN;

   private:

      // Theory flag
      enum STUType { kSTU = 1, kLittlestHiggs, kRandallSundrum, kRandallSundrum_CS, kTechniColor, 
                     kExtraFamilies, kSUSY,  kIDM, kUED, kTwoHDM };

      STUType m_replace;
      
      Bool_t IsSTU()               const { return (m_replace == kSTU)               ? kTRUE : kFALSE; }
      Bool_t IsLittlestHiggs()     const { return (m_replace == kLittlestHiggs)     ? kTRUE : kFALSE; }
      Bool_t IsRandallSundrum()    const { return (m_replace == kRandallSundrum)    ? kTRUE : kFALSE; }
      Bool_t IsRandallSundrum_CS() const { return (m_replace == kRandallSundrum_CS) ? kTRUE : kFALSE; }
      Bool_t IsTechniColor()       const { return (m_replace == kTechniColor)       ? kTRUE : kFALSE; }
      Bool_t IsExtraFamilies()     const { return (m_replace == kExtraFamilies)     ? kTRUE : kFALSE; }  
      Bool_t IsSUSY()              const { return (m_replace == kSUSY)              ? kTRUE : kFALSE; }
      Bool_t IsUED()               const { return (m_replace == kUED)               ? kTRUE : kFALSE; }  
      Bool_t IsInertDoublet()      const { return (m_replace == kIDM)               ? kTRUE : kFALSE; }  
      Bool_t IsTwoHDM()            const { return (m_replace == kTwoHDM)            ? kTRUE : kFALSE; }

      Bool_t AddSM;

      ClassDef(TheoryHandler,0)

   };
}

#endif
