/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : UED                                                                   *
 *                                                                                *
 * Description:                                                                   *
 *      implementation                                                            *
 *                                                                                *
 * Sources:                                                                       *
 *     - I. Gogoladze and C. Macesanu,                                            *
 *       Phys. Rev.  D 74, 093012 (2006), [arXiv:hep-ph/0605207]                  *
 *     - consult also:                                                            *      
 *       D. Hooper and S. Profumo,                                                *
 *       Phys. Rept. 453, 29 (2007), [arXiv:hep-ph/0701197]                       *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"

#include "GSTU/UED.h"

using namespace Gfitter;

ClassImp(GSTU::UED)

GSTU::UED::UED()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // book SM parameters
   BookParameter( "MH"            ,&p_MH );
   BookParameter( "mt"            ,&p_mt       ); 
   BookTheory   ( "GEW::MW"       ,& t_MW );

   // needed for sw2
   BookParameter( "MZ"            ,&p_MZ       );

   // book UED parameters
   BookParameter( "MKK"          ,&p_MKK         ); 
}


void GSTU::UED::Initialise(){}

// eq. (12) from hep-ph/0605207
Double_t GSTU::UED::GetS()
{
   Double_t MW  = GetMW().GetTheoryPrediction();
   Double_t cw2 = TMath::Power( MW/p_MZ, 2 );
   Double_t sw2 = 1. - cw2;
   Double_t vev = GConstants::vev();
   Double_t g   = 2*MW/vev;

   Double_t Shat1 = TMath::Power( g* p_mt, 2 ) / ( 576.   * TMath::Power( p_MKK, 2 ) );
   Double_t Shat2 = TMath::Power( g* p_MH, 2 ) / ( 2304.  * TMath::Power( p_MKK, 2 ) );

   Double_t Shat = Shat1 + Shat2;

   return ( 4 * sw2 * Shat ) / GConstants::alphaQED() ;
}

// eq. (12) from hep-ph/0605207
Double_t GSTU::UED::GetU()
{
   Double_t pi2  = TMath::Power( TMath::Pi(), 2) ;
   Double_t MW  = GetMW().GetTheoryPrediction() ;
   Double_t cw2 = TMath::Power( MW/p_MZ, 2 );
   Double_t sw2 = 1. - cw2;
   Double_t vev = GConstants::vev();
   Double_t g   = 2*MW/vev;

   Double_t Uhat1 = sw2 * TMath::Power( g * MW, 2 ) / ( 576.  * TMath::Power( p_MKK, 2 ) );
   Double_t Uhat2 = sw2 * pi2 * TMath::Power( g * MW * p_MH , 2 ) / ( 21600. * TMath::Power( p_MKK, 4 ) );

   Double_t Uhat = Uhat1 - Uhat2;

   return (-4 * sw2 * Uhat) / GConstants::alphaQED() ;
}

// eq. (12) from hep-ph/0605207
Double_t GSTU::UED::GetT()
{

   Double_t MW  = GetMW().GetTheoryPrediction();
   Double_t cw2 = TMath::Power( MW/p_MZ, 2 );
   Double_t sw2 = 1. - cw2;
   Double_t vev = GConstants::vev();
   Double_t g   = 2*MW/vev;

   Double_t That1 = g * g * TMath::Power( p_mt, 4 ) / ( 96. * TMath::Power( MW * p_MKK, 2 ) ) ;
   Double_t That2 = 5 * sw2 * TMath::Power( g * p_MH, 2 ) / ( 1152. * cw2 * TMath::Power( p_MKK, 2 ) ) ;

   Double_t That = That1 - That2;   

   return That / GConstants::alphaQED();

}

