/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GContour                                                              *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>
#include "TAxis.h"
#include "TH1F.h"
#include "TH2F.h"
#include "TTree.h"
#include "TGraph.h"
#include "Gfitter/GContour.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GUtils.h"
#include "Gfitter/GMinuitFitter.h"
#include "Gfitter/GFitterBase.h"
#include "Gfitter/GTimer.h"
#include "Gfitter/GStringList.h"
#include "Gfitter/GResultScan1d.h"

using namespace std;

Gfitter::GContour::GContour()
{
   InitClassName( "GContour" );
}

Gfitter::GContour::~GContour()
{}

void Gfitter::GContour::Contour2D( const TString& par1, const TString& par2, const std::vector<TString>& options )
{
   GParameter* gpar1 = gStore()->GetParameter( par1 );
   GParameter* gpar2 = gStore()->GetParameter( par2 );

   // sanity checks
   if (gpar1 == 0) m_logger << kFATAL << "<Contour2D> Fatal error: unknown parameter \"" << par1 << "\": "
                            << " check datacard" << GEndl;
   if (gpar2 == 0) m_logger << kFATAL << "<Contour2D> Fatal error: unknown parameter \"" << par2 << "\": "
                            << " check datacard" << GEndl;

   Contour2D( gpar1, gpar2, options );
}

void Gfitter::GContour::Contour2D( GParameter* gpar1, GParameter* gpar2, const std::vector<TString>& options )
{
   // sanity check
   if (gpar1 == 0 || gpar2 == 0) m_logger << kFATAL << "<Contour2D> Zero GParameter pointer: " 
                                          << gpar1 << " or " << gpar2 << GEndl;

   m_logger << kINFO << "Scanning contours for parameters \"" << gpar2->GetParName() 
            << "\" versus \"" << gpar1->GetParName() << "\"" 
            << " in the intervals [" << gpar2->GetScanRange().GetMin() << ", " 
            << gpar2->GetScanRange().GetMax() << "] and ["<< gpar1->GetScanRange().GetMin() << ", " 
            << gpar1->GetScanRange().GetMax() << "]"
            << GEndl;

   if (options.size() != 2) m_logger << kFATAL 
                                     << "Expected 2 parameters (DeltaChi2 and NPoints) to be "
                                     << "set in options" << GEndl;

   std::vector<Double_t> dchi2;
   TString               dchi2txt;
   Int_t                 npoints = -1;
   for (UInt_t iarg=0; iarg<options.size(); iarg++) {      
      if (options[iarg].Contains( "DeltaChi2" )) {
         TString par = GUtils::ExtractValueFromParSetting( options[iarg], m_logger );

         // now the syntax can be "1;2;3;..."
         GStringList ll( par, ";" );
         for (Int_t jarg=0; jarg<ll.GetEntries(); jarg++) {
            dchi2.push_back( ll.GetArg( jarg ).Atof() );
            dchi2txt += Form( "%g", dchi2.back() );
            if (jarg+1 < ll.GetEntries()) dchi2txt += ", ";
         }
      }         
      else if (options[iarg].Contains( "NPoints" )) {
         npoints = GUtils::ExtractValueFromParSetting( options[iarg], m_logger ).Atoi();
      }
   }

   // sanity checks
   if (npoints < 0) m_logger << kFATAL 
                             << "Mismatch in option string: could not "
                             << "find non-zero number of points to use in contour (parameter \"NPoints\")" 
                             << GEndl;
         
   if (dchi2.size() < 1) m_logger << kFATAL 
                                  << "Do not know for which dchi2 to draw contours, please check data card" 
                                  << GEndl;

   m_logger << kINFO << "Determine contour(s) for Delta_chi2(s): " << dchi2txt << GEndl;
   m_logger << kINFO << "Number of contour points              : " << npoints << GEndl;

   // first execute the global fit
   gStore()->GetFitter()->Initialise();
   gStore()->GetFitter()->ExecuteFit();

   // now do the contour (iterate over all delta-chi2)
   Double_t* npx = new Double_t[npoints+1];
   Double_t* npy = new Double_t[npoints+1];
   std::vector<Double_t>::iterator it = dchi2.begin();
   // DeltaChi2 for different coverage probabilities:
   // dchi2 = 1    -> Prob = 39.35%
   // dchi2 = 2.3  -> Prob = 68.27% (1 sigma)
   // dchi2 = 4.61 -> Prob = 90%
   // dchi2 = 5.99 -> Prob = 95%
   // dchi2 = 6.18 -> Prob = 95.45% (2 sigma)
   // dchi2 = 9.21 -> Prob = 99%
   // dchi2 =11.83 -> Prob = 99.73% (3 sigma)
   for (; it != dchi2.end(); it++) {
      gStore()->GetFitter()->DrawContour( gpar1, gpar2, npoints, *it, npx, npy );

      // close the contour
      npx[npoints] = npx[0];
      npy[npoints] = npy[0];

      // create graph for contour
      TGraph* grdchi2 = new TGraph( npoints+1, npx, npy );

      // set the graph's name
      TString grname = Form( "%s_vs_%s_%g", 
                             gpar2->GetParNameWONSpace().Data(), gpar1->GetParNameWONSpace().Data(), *it );
      grdchi2->SetNameTitle( grname, grname );

      // save to output stream
      grdchi2->Write();
   }
   
   delete [] npx;
   delete [] npy;
}

