/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GController                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <map>
#include <TTree.h>
#include "Riostream.h"
#include "Gfitter/GController.h"
#include "Gfitter/GDCardInterpreterXML.h"
#include "Gfitter/GResultWriterScreen.h"
#include "Gfitter/GResultWriterHtml.h"
#include "Gfitter/GResultWriterLatex.h"
#include "Gfitter/GResultWriterRoot.h"
#include "Gfitter/GResultBase.h"
#include "Gfitter/GResultEval.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GFunctionOfFreeParameter.h"
#include "Gfitter/GScan.h"
#include "Gfitter/GScanAnalysis.h"
#include "Gfitter/GToyAnalysis.h"
#include "Gfitter/GContour.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryFactory.h"
#include "Gfitter/GFitterBase.h"
#include "Gfitter/GMinuitFitter.h"
#include "Gfitter/GGeneticFitter.h"
#include "Gfitter/GSimulatedAnnealing.h"
#include "Gfitter/GGSLMinimizer.h"
#include "Gfitter/GRandomSimplex.h"
#include "Gfitter/GRootClassFactory.h"
#include "Gfitter/GAction.h"
#include "TROOT.h"

ClassImp(Gfitter::GController)

Gfitter::GController::GController()
   : m_dataCardInterpreter( 0 ),
     m_targetFile( 0 )
{
   InitClassName( "GController" );
}

Gfitter::GController::~GController()
{}

void Gfitter::GController::InterpretDataCard( const TString& dataCard, 
                                              GDCardInterpreterBase::DataCardType dataCardType )
{
   // create DCardInterpreter
   switch (dataCardType) {
   case GDCardInterpreterBase::XML:
      m_dataCardInterpreter = new GDCardInterpreterXML( dataCard );
      break;
   default:
      m_logger << kFATAL << "Unknown dataCard type: " << dataCardType << GEndl;
   }
   
   // interpret the data card
   m_dataCardInterpreter->Interpret();
}

void Gfitter::GController::Initialise()
{
   // first complete initialisation of all GParameters
   gStore()->InitAllParameters();

   // do we use parameter caching to speed up fitting
   if (gStore()->ExistVariable( "GfitterFlags::UseParCaching" ) && 
       gStore()->GetVariable( "GfitterFlags::UseParCaching" )->GetBoolValue()) {
      m_logger << kINFO << "Parameter caching for theory recomputation is ENABLED" << GEndl;      
   }
   else {
      m_logger << kINFO << "Parameter caching for theory recomputation is NOT ENABLED" << GEndl;      
   }

   // create target file (default is "gfitter.root")
   const TString& target = ( gStore()->ExistVariable( "Files::GraphicsOut" ) ?
                             gStore()->GetVariable( "Files::GraphicsOut" )->GetStringValue() : "gfitter.root" );
   m_logger << kINFO << "<Initialise> Create target file: \"" << target << "\"" << GEndl;
   gStore()->SetTargetRootFile( m_targetFile = TFile::Open( target, "RECREATE" ) );         
   
	// initialise the result writer (expected format is "
   TString resultFormat = ( gStore()->ExistVariable( "Files::ResultFormat" ) ?
                            gStore()->GetVariable( "Files::ResultFormat" )->GetStringValue() : "" );
   resultFormat.ToLower();

	// html or latex ?
   m_resultWriters.clear();
   // add always screen
   m_resultWriters.push_back( new GResultWriterScreen() );

   TString restarget = target; restarget.ReplaceAll(".root","");
	  if (resultFormat.Contains( "html"  )) {
      m_resultWriters.push_back( new GResultWriterHtml ( restarget + ".html" ) );
      m_logger << kINFO << "<Initialise> Create result file: \"" 
               << m_resultWriters.back()->GetResultFileName() << "\"" << GEndl;
   }
   if (resultFormat.Contains( "latex" )) {
      m_resultWriters.push_back( new GResultWriterLatex( restarget + ".tex" ) );
      m_logger << kINFO << "<Initialise> Create result file: \"" 
               << m_resultWriters.back()->GetResultFileName() << "\"" << GEndl;
   }
   if (resultFormat.Contains( "root" )) {
      m_resultWriters.push_back( new GResultWriterRoot() );
      m_logger << kINFO << "<Initialise> Will write results into ROOT target file" << GEndl;
   }

   // init all result writers
   std::vector<GResultWriterBase*>::iterator it = m_resultWriters.begin();
   for (; it != m_resultWriters.end(); it++) (*it)->Initialise();

   // initialise the parameters that have theories
   if (!GTheoryFactory::InitTheories()) m_logger << kFATAL << "<Initialise> InitTheories failed !" << GEndl;
   
   // create the fitter (default is "Minuit")
   const TString& fitterName = ( gStore()->ExistVariable( "Fitter::FitterType" ) ? 
                                 gStore()->GetVariable( "Fitter::FitterType" )->GetStringValue() : "Minuit" );
   TString myFitterName = fitterName.Data();
   myFitterName.ReplaceAll( " ", "" );
   GFitterBase* fitter( 0 );
   if (myFitterName[0] == 'T' || myFitterName[0] == 't') {
      myFitterName.Replace( 0, 2, "", 0 );
      TObjArray* fitArr = myFitterName.Tokenize(":");
      TObjString* objstring = (TObjString*)fitArr->At(0);
      TString fitName(objstring->GetString()); fitName.ToLower();
      if (fitName == "minuit") { 
        fitter = new GMinuitFitter();
      } else if (fitName == "gslminimizer") {
         if (fitArr->GetEntries()==2) {
           TString myType = ((TObjString*)fitArr->At(1))->GetString();
           fitter = new GGSLMinimizer(myType.Data());
         } else if (fitArr->GetEntries()>=3) {
           TString myType = ((TObjString*)fitArr->At(1))->GetString();
           TString myAlg  = ((TObjString*)fitArr->At(2))->GetString();
           fitter = new GGSLMinimizer(myType.Data(),myAlg.Data());
         } else { fitter = new GGSLMinimizer(); } // Minuit2, migrad
      } else { // nothing else implemented at present
         m_logger << kFATAL <<  "<Initialise> Unknown fitter: \"" << fitterName << "\"" << GEndl;
      }
      delete fitArr;
   }
   else if (!myFitterName[0] == 'f') {
      m_logger << kFATAL << "<Initialise> Mismatch in \"Fitter::FitterType\" field. First entry "
               << "must be a boolean \"T/F\"" << GEndl;
   }
   
   // upload fitter to GStore
   gStore()->SetFitter( fitter );

   // create the prefitter (if required)
   // format is "T:Genetic", ie, the first boolean determines whether a prefitter is used
   const TString& preFitterName = ( gStore()->ExistVariable( "Fitter::PreFitterType" ) ? 
                                    gStore()->GetVariable( "Fitter::PreFitterType" )->GetStringValue() : "" );
   TString myPreFitterName = preFitterName.Data();
   myPreFitterName.ToLower(); myPreFitterName.ReplaceAll( " ", "" );
   GFitterBase* preFitter( 0 );
   if (myPreFitterName[0] == 't') {
      myPreFitterName.ReplaceAll( "t:", "" );
      if (myPreFitterName == "genetic") preFitter = new GGeneticFitter();
      else if (myPreFitterName == "annealing") preFitter = new GSimulatedAnnealing();
      else if (myPreFitterName == "randomsimplex") preFitter = new GRandomSimplex();
      else { // nothing else implemented at present
         m_logger << kFATAL <<  "<Initialise> Unknown prefitter: \"" << preFitterName << "\"" << GEndl;
      }
   }
   else if (!myPreFitterName[0] == 'f') {
      m_logger << kFATAL << "<Initialise> Mismatch in \"Fitter::PreFitterType\" field. First entry "
               << "must be a boolean \"T/F\"" << GEndl;
   }
   
   // upload fitter to GStore
   gStore()->SetPreFitter( preFitter );

   //upload the penalty factor to GStore
   Double_t penaltyFactor       = (gStore()->ExistVariable( "Fitter::PenaltyFactor" ) ? 
                                   gStore()->GetVariable( "Fitter::PenaltyFactor" )->GetFloatValue() : 0.15);
   gStore()->SetPenaltyFactor( penaltyFactor );

   // set random seed, default is zero = cpu clock
   if ( gStore()->ExistVariable("GfitterFlags::RandomSeed") ) {
     int randomSeed = gStore()->GetVariable( "GfitterFlags::RandomSeed" )->GetIntValue() ;
     this->SetRandomSeed( randomSeed );
   }

   // finally (at the very end !) dump the GStore parameters
   gStore()->DumpParameters();
}

void Gfitter::GController::ExecuteAllActions() 
{
   // iterate over all actions
   std::vector<const GAction*>::const_iterator action = gStore()->GetActions().begin();
   for (; action != gStore()->GetActions().end(); action++) {

      if (!(*action)->IsActive()) {
         m_logger << kINFO << "Inactive action: \"" << (*action)->GetActionName() 
                  << "\" will not be executed" << GEndl;
         continue;
      }
      
      m_logger << kINFO << "Execute action: \"" << (*action)->GetActionName() << "\"" << GEndl;

      // retrieve arguments and parameters of action
      m_args = (*action)->GetArgs();
      m_pars = (*action)->GetPars();

      // name of action
      TString lowerActionName = (*action)->GetActionName();
      lowerActionName.ToLower();
            
      // perform actions
      if      (lowerActionName == "evaluate"      ) EvaluateAction();
      else if (lowerActionName == "evaluatetoy"   ) EvaluateToyAction();
      else if (lowerActionName == "contour2d"     ) Contour2DAction();
      else if (lowerActionName == "fctoffreepara" ) FctOfFreeParaAction();
      else if (lowerActionName == "scan1d"        ) Scan1DAction();
      else if (lowerActionName == "scananalysis"  ) ScanAnalysisAction();
      else if (lowerActionName == "scan2d"        ) Scan2DAction();
      else if (lowerActionName == "toyscan1d"     ) ToyScan1DAction();
      else if (lowerActionName == "toyscan2d"     ) ToyScan2DAction();
      else if (lowerActionName == "validate"      ) ValidateAction();
      else {
         m_logger << kFATAL << "Unknown action: \"" << lowerActionName << "\"" << GEndl;
      }
   }
}

void  Gfitter::GController::UpdateResultWriters( const TString& actionName )
{
   // write results
   std::vector<GResultWriterBase*>::iterator resit = m_resultWriters.begin();
   for (; resit != m_resultWriters.end(); resit++) (*resit)->AddResults( actionName );
}

void Gfitter::GController::Terminate() 
{
   // write results
   m_logger << kINFO << "Terminate Job..." << GEndl;
   std::vector<GResultWriterBase*>::iterator it = m_resultWriters.begin();
   for (; it != m_resultWriters.end(); it++) {
      (*it)->Finalise();
      if ((*it)->GetResultFileName() != "") 
         m_logger << kINFO << "Wrote result file: " << (*it)->GetResultFileName() << GEndl;
   }

   // close the ROOT target file
   m_targetFile->Close();
   m_logger << kINFO << "Wrote output file: " << m_targetFile->GetName() << GEndl;
}

// -------------- Actions ---------------------------------------------

void Gfitter::GController::EvaluateAction()
{
   // global fit
   gStore()->GetFitter()->Initialise();
   Double_t chi2 = gStore()->GetFitter()->ExecuteFit();
   gStore()->GetFitter()->UpdateResults();

   // store fit results in a tree
   TTree* FitResults = new TTree( "FitResults", "FitResults" );
    
   // start value tree
   TTree* InitValues = new TTree( "InitValues", "Initial Values of parameters" );

   // how many active parameters
   Int_t num = 0 ;
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
   for ( ;par != gStore()->GetActiveParameters().end(); par++) num++;

   Double_t fitval[num];
   Double_t InitVal[num];
   Double_t InitErrp[num];
   Double_t InitErrm[num];
   
   Int_t i = 0;

   // create the branches and get the fit results   
   for (par = gStore()->GetActiveParameters().begin(); par != gStore()->GetActiveParameters().end(); par++) {
      // init values
      InitValues->Branch( (*par)->GetFullNameWONSpace() + (TString)"_InitVal" ,  &InitVal[i], "InitVal/D" );
      InitValues->Branch( (*par)->GetFullNameWONSpace() + (TString)"_InitErrp" , &InitErrp[i], "InitErrp/D" );
      InitValues->Branch( (*par)->GetFullNameWONSpace() + (TString)"_InitErrm" , &InitErrm[i], "InitErrm/D" );
      InitVal[i]  =  (*par)->GetValue();
      InitErrp[i] = +(*par)->GetErrGaussp();
      InitErrm[i] = -(*par)->GetErrGaussm();
      
      // fit values
      FitResults->Branch( (*par)->GetFullNameWONSpace() , &fitval[i], "fitval/D" );
      if ((*par)->HasTheory() ) fitval[i] = (*par)->GetTheoryPrediction();
      else                      fitval[i] = (*par)->GetFitValue();
      i++;
   }
   InitValues->Fill();
   FitResults->Branch("chi2" , &chi2, "chi2/D" );
   FitResults->Fill();
  
   i = 0;
   for (par = gStore()->GetActiveParameters().begin(); par != gStore()->GetActiveParameters().end(); par++) {
      if ((*par)->HasTheory() || (*par)->IsCurve() || !(*par)->IsFirst() ) fitval[i] = 0;
      else fitval[i] = (*par)->GetResult()->GetErrGaussAsym(1);
      i++;
   }
   FitResults->Fill();
   
   i = 0;
   for (par = gStore()->GetActiveParameters().begin(); par != gStore()->GetActiveParameters().end(); par++) {
      if ((*par)->HasTheory() || (*par)->IsCurve() || !(*par)->IsFirst() ) fitval[i] = 0;
      else fitval[i] = (*par)->GetResult()->GetErrGaussAsym(-1);
      i++;
   }
   FitResults->Fill();
   
   InitValues->Write();
   FitResults->Write();
   delete FitResults;
   delete InitValues;

   // update result writer
   UpdateResultWriters( "Evaluate" );
}

void Gfitter::GController::EvaluateToyAction()
{
   //gRandom->SetSeed(0);

   // toys MC generation of of global fits
   gStore()->GetFitter()->Initialise();
   Double_t chi2min = gStore()->GetFitter()->ExecuteFit();
   gStore()->GetFitter()->UpdateResults();

   // modify intitial values, e.g. for coverage test
   const GParPtrVec_t& v = gStore()->GetActiveParameters();
   GParPtrVec_t::const_iterator par;
   for (par = v.begin(); par != v.end(); par++) {
      if ( !(*par)->HasTheory() && (*par)->GetErrGaussp() != 0 && (*par)->GetModifyInToy() != 0) {
         m_logger << kINFO << "Parameter " << (*par)->GetParName() << " is set from "
                  << (*par)->GetFitValue()
                  << " to " << (*par)->GetFitValue() + (*par)->GetModifyInToy() * (*par)->GetErrGaussp() << GEndl;
         (*par)->SetFitValue( (*par)->GetFitValue() + (*par)->GetModifyInToy() * (*par)->GetErrGaussp() );
      }
   }   
   Double_t chi2minModify = gStore()->GetFitter()->GetCurrentChi2();

   // interpret parameters: must set the number of toy experiments to be performed
   Int_t nToyExp = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "NToyExp" )) {
         nToyExp = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
         break;
      }
   }
   if (nToyExp <= 0) m_logger << kFATAL 
                              << "<EvaluateToyAction> Mismatch in \"EvaluateToy\": cannot "
                              << "find non-zero number of toys to generate (parameter \"NToyExp\")" 
                              << GEndl;
   
   GToyAnalysis* ta = new GToyAnalysis();
   ta->ExecuteToy( chi2minModify, nToyExp );
   delete ta;
   if( chi2minModify != chi2min ){
      m_logger << kINFO << "Due to the modification of the intitial model parameters,"
               << "the chi2min has changed from " << chi2min << " to "<< chi2minModify << GEndl;
   }   
}

void Gfitter::GController::Contour2DAction()
{
   // create 2D contours of parameters (w/o theory)
   if (m_args.size() != 2) m_logger << kFATAL 
                                    << "<Contour2DAction> wrong number of arguments in action string "
                                    << "for 2D contour: " << m_args.size() << GEndl;
   
   GContour* contour = new GContour();
   contour->Contour2D( m_args[0], m_args[1], m_pars );
   delete contour;
}

void Gfitter::GController::Scan1DAction()
{
   // scan parameter
   GScan* scanner = new GScan();
   
   // extract required numbers of bins
   Int_t nbins = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "Nbins" )) {
               nbins = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
               break;
      }
   }
   if (nbins <= 0) m_logger << kFATAL 
                            << "<Scan1DAction> Mismatch in \"Scan\": cannot "
                            << "find non-zero number of bins to use in 1D scan (parameter \"Nbins\")" 
                            << GEndl;
   
   // loop over parameters for this action
   for (it = m_args.begin(); it != m_args.end(); it++) {
      m_logger << kINFO << "The 1D-scan variable is: \"" << (*it) << "\"" << GEndl;
      scanner->Scan1D( (*it), nbins );
      
      // update result writer
      UpdateResultWriters( "Scan1D" );
   }
   
   delete scanner;
}

void Gfitter::GController::Scan2DAction()
{
   // scan parameter
   GScan* scanner = new GScan();
   
   // extract required numbers of bins
   Int_t nbins1 = -1;
   Int_t nbins2 = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "Nbins" )) {
         nbins1 = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
         nbins2 = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
         break;
      }
      else if (it->Contains( "N1bins" )) { 
         nbins1 = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
      else if (it->Contains( "N2bins" )) { 
         nbins2 = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
   }
   if (nbins1 <= 0 && nbins2 <= 0) m_logger << kFATAL 
                            << "<Scan2DAction> Mismatch in \"Scan\": cannot "
                            << "find non-zero number of bins to use in 1D scan (parameter \"Nbins\")" 
                            << GEndl;
   
   // loop over parameters for this action
   if (m_args.size() != 2) m_logger << kFATAL 
                                  << "<Scan2DAction> wrong number of arguments in action string "
                                  << "for 2D scan: " << m_args.size() << GEndl;
   
   m_logger << kINFO << "The 2D-scan variables are: \"" 
            << m_args[0] << "\" with "<< nbins1 << " bins " 
            << "and \"" << m_args[1] << "\" with " << nbins2 << " bins" << GEndl;
   scanner->Scan2D( m_args[0], m_args[1], nbins1, nbins2 );
   
   delete scanner;
}

void Gfitter::GController::ScanAnalysisAction()
{
   // scan parameter
   GScanAnalysis* scanAnalysis = new GScanAnalysis();
   
   // extract required numbers of bins
   Int_t nbins = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "Nbins" )) {
               nbins = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
               break;
      }
   }
   if (nbins <= 0) m_logger << kFATAL 
                            << "<ScanAnalysisAction> Mismatch in \"Scan\": cannot "
                            << "find non-zero number of bins to use in 1D scan (parameter \"Nbins\")" 
                            << GEndl;
   
   m_logger << kINFO << "The 1D-scan variable is: \"" << m_args[0] << "\"" << GEndl;
   scanAnalysis->ExecuteScan( m_args[0], nbins );

   delete scanAnalysis;
}

void Gfitter::GController::ValidateAction() 
{
   // validate theory setup
   gStore()->GetFitter()->Initialise();
   m_logger << kINFO << "-------------------------------------------------" << GEndl;
   
   cout << "Active parameters: " << gStore()->GetActiveParameters().size() << endl;
   const GParPtrVec_t& paramsAct = gStore()->GetActiveParameters();
   GParPtrVec_t::const_iterator parIt = paramsAct.begin();
   for (; parIt != paramsAct.end(); parIt++) {
      cout << "   " << (*parIt)->GetFullName() << ", has " << ((*parIt)->HasTheory()?"a":"no") << " theory" << endl;
      if ((*parIt)->HasTheory()) {
         const GTheory* parTheory = (*parIt)->GetTheory();
         parTheory->Cout(cout,2);
      }
   }
   
   cout << "Fit parameters: " << gStore()->GetFitter()->GetNumOfFitPars() << endl;
   const GParPtrVec_t& params = gStore()->GetFitter()->GetFitPars();
   parIt = params.begin();
   for (; parIt!=params.end(); parIt++) {
      cout << "   " << (*parIt)->GetFullName() << ", has " << ((*parIt)->HasTheory()?"a":"no") << " theory" << endl;
   }
   m_logger << kINFO << "-------------------------------------------------" << GEndl;
}

void Gfitter::GController::ToyScan1DAction()
{
   // scan parameter
   GToyAnalysis* toyScan = new GToyAnalysis();
   
   // extract required numbers of bins
   Int_t nbins = -1;
   Int_t ntoys = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "Nbins" )) {
               nbins = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
      if (it->Contains( "Ntoys" )) {
               ntoys = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
   }
 
   if (nbins <= 0) m_logger << kFATAL 
                            << "<ToyScan1DAction> Mismatch in \"ToyScan\": cannot "
                            << "find non-zero number of bins to use in 1d toy scan (parameter \"Nbins\")" 
                            << GEndl;
   
   if (ntoys <= 0) m_logger << kFATAL 
                            << "<ToyScan1DAction> Mismatch in \"ToyScan\": cannot "
                            << "find non-zero number of toys to use in 1d toy scan (parameter \"Ntoys\")" 
                            << GEndl;
   

     
   m_logger << kINFO << "The 1D-scan variable is: \"" << m_args[0] << "\"" << GEndl;
   toyScan->Scan1DToy( m_args[0], nbins, ntoys );
   
   delete toyScan;
}

void Gfitter::GController::ToyScan2DAction()
{
   // scan parameter
   GToyAnalysis* toyScan = new GToyAnalysis();
   
   // extract required numbers of bins
   Int_t nbinsX = -1;
   Int_t nbinsY = -1;
   Int_t ntoys  = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "NbinsX" )) {
         nbinsX = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
      if (it->Contains( "NbinsY" )) {
         nbinsY = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
      if (it->Contains( "Ntoys" )) {
         ntoys = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
      }
   }
   if (nbinsX <= 0) m_logger << kFATAL 
                            << "<ToyScan2DAction> Mismatch in \"Scan\": cannot "
                            << "find non-zero number of bins to use in 2D toy scan (parameter \"NbinsX\")" 
                            << GEndl;
   if (nbinsY <= 0) m_logger << kFATAL 
                            << "<ToyScan2DAction> Mismatch in \"Scan\": cannot "
                            << "find non-zero number of bins to use in 2D toy scan (parameter \"NbinsY\")" 
                            << GEndl;
   if (ntoys <= 0) m_logger << kFATAL 
                            << "<ToyScan2DAction> Mismatch in \"Scan\": cannot "
                            << "find non-zero number of toys to use in 2D toy scan (parameter \"Nbins\")" 
                            << GEndl;
      
   // loop over parameters for this action
   if (m_args.size() != 2) m_logger << kFATAL 
                                  << "<ToyScan2DAction> wrong number of arguments in action string "
                                  << "for 2D scan: " << m_args.size() << GEndl;
   
   m_logger << kINFO << "The 2D-scan variables are: \"" 
            << m_args[0] << "\" and \"" << m_args[1] << "\"" << GEndl;
   
   toyScan->Scan2DToy( m_args[0], m_args[1], nbinsX, nbinsY, ntoys );
   
   delete toyScan;
}

void Gfitter::GController::FctOfFreeParaAction()
{
   // scan parameter
   GFunctionOfFreeParameter* scanner = new GFunctionOfFreeParameter();
   
   // extract required numbers of bins
   Int_t nbins = -1;
   std::vector<TString>::const_iterator it = m_pars.begin();
   for (; it != m_pars.end(); it++) {
      if (it->Contains( "Nbins" )) {
               nbins = GUtils::ExtractValueFromParSetting( *it, m_logger ).Atoi();
               break;
      }
   }
   if (nbins <= 0) m_logger << kFATAL 
                            << "<FctOfFreeParaAction> Mismatch in \"FctOfFreePara\": cannot "
                            << "find non-zero number of bins (parameter \"Nbins\")" 
                            << GEndl;
   
   // loop over parameters for this action
   for (it = m_args.begin(); it != m_args.end(); it++) {   
      m_logger << kINFO << "The free floating variable is: \"" << (*it) << "\"" << GEndl;
      scanner->FctOfFreePara( (*it), nbins );   
   }

   delete scanner;
}

void Gfitter::GController::Reset() 
{
  gStore()->ClearStore();
}

void Gfitter::GController::SetExternalParameter(const char* name, double value, bool active )
{
  if (name==0) return;
  Gfitter::GParameter* par  = gStore()->GetParameter( name );
  if (par!=0) { 
    if ( !par->HasTheory() ) par->SetFitValue(value);
    else par->SetValue(value);
    par->SetActive(active);
    m_logger << kINFO << "Parameter <" << name << "> now set to value = " << value << ", active = " << active << GEndl;
  } else { 
    m_logger << kWARNING << "Could not set parameter <" << name << ">. Unknown." << GEndl; 
  }
}

void Gfitter::GController::SetRandomSeed( int randomSeed )
{
  if ( randomSeed < 0 ) return;
  gStore()->SetVariable( "GfitterFlags::RandomSeed", Form("%d",randomSeed) );
  Gfitter::GObject::SetRandomSeed( randomSeed );
  m_logger << kINFO << "Random seed now set to value = " << randomSeed << GEndl;
}

void Gfitter::GController::SetGraphicsOut( const char* name )
{
  if (name==0) return;
  gStore()->SetVariable( "Files::GraphicsOut", name );
  m_logger << kINFO << "Output file now set to = " << name << GEndl;  
}

