/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GController                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Controls all fit steps                                                    *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GController
#define Gfitter_GController

#include "TFile.h"

#include "Gfitter/GObject.h"
#include "Gfitter/GDCardInterpreterBase.h"
#include "Gfitter/GResultWriterBase.h"

class TXMLNode;

namespace Gfitter {
   
   class GController : public GObject {
      
   public:
      
      GController();
      virtual ~GController();

      void InterpretDataCard( const TString& dataCard, 
                              GDCardInterpreterBase::DataCardType dataCardType = GDCardInterpreterBase::XML );

      // called after interpretation of datacard
      void Initialise();

      // called in between initialise and terminate
      void ExecuteAllActions();

      // must be called at end of job
      void Terminate();

      // retrieve output file
      TFile* GetTargetFile() const { return m_targetFile; }

      // MB: Reset, needed for several Gfitter jobs in one root session
      void Reset();

      // MB: external method to set parameter
      void SetExternalParameter( const char* name=0, double value=0, bool active=true );
      // MB: Set random seed externally
      void SetRandomSeed( int randomSeed=0 );
      // MB : Set output file externally
      void SetGraphicsOut( const char* name=0 ); 
      
   private:

      // interpretation of data card
      GDCardInterpreterBase* m_dataCardInterpreter;

      // --- actions
      
      void EvaluateAction();         // global fit
      void EvaluateToyAction();      // toys MC generation of of global fits
      void Contour2DAction();        // create 2D contours of parameters (w/o theory)
      void FctOfFreeParaAction();    // compute theories in dependency of free parameter (no fit)
      void Scan1DAction();           // parameter scans (1D)
      void Scan2DAction();           // parameter scans (2D)
      void ScanAnalysisAction();     // scan analysis (1D)
      void ToyScan1DAction();        // parameter scan (1D), CL is computed via toy experiments
      void ToyScan2DAction();        // parameter scan (2D), CL is computed via toy experiments
      void ValidateAction();         // validate the setup of all theories

      // update result writers
      void UpdateResultWriters( const TString& actionName );

      // helper members for action interpretation
      std::vector<TString>      m_args;
      std::vector<TString>      m_pars;

      // writing of results
      std::vector<GResultWriterBase*> m_resultWriters;

      // global jobname
      TString      m_jobName;

      // target output file
      TFile*       m_targetFile;

      ClassDef(GController,1)
   };
}

#endif
