/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GCorrGaussGen                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include "TArrayD.h"
#include "TMath.h"
#include "TRandom.h"

#include "Gfitter/GCorrGaussGen.h"

ClassImp(Gfitter::GCorrGaussGen)

Gfitter::GCorrGaussGen::GCorrGaussGen( const TMatrixD* covMat )
   : GObject(),
     m_size( 0 ),
     m_sqrMat( 0 )
{
   // standard constructor
   InitClassName( "GCorrGaussGen" );

   if (covMat != 0) {
      m_size = covMat->GetNrows();
      m_sqrMat = new TMatrixD( m_size, m_size );
      Initialize( *covMat );
   }
}

Gfitter::GCorrGaussGen::GCorrGaussGen( const TMatrixD& covMat )
   : GObject(),
     m_size( covMat.GetNrows() ),
     m_sqrMat( new TMatrixD( m_size, m_size ) )
{
   // standard constructor
   InitClassName( "GCorrGaussGen" );

   // sanity checks, matrix must be quadratic and symmetric
   if (covMat.GetNrows() != covMat.GetNcols()) 
      m_logger << kFATAL << "input matrix is not quadratic: " 
               << covMat.GetNrows() << " " << covMat.GetNcols() << GEndl;
   for (Int_t i=0;i< m_size; i++) {
      for (Int_t j=0; j<i; j++) {
         if (covMat(i,j) != covMat(j,i)) 
            m_logger << kFATAL << "input matrix is not symmetric: " 
                     << covMat(i,j) << " " << covMat(j,i)
                     << " in bins: " << i << " " << j << GEndl;
      }
   }   

   this->Initialize( covMat );
}

void Gfitter::GCorrGaussGen::Initialize( const TMatrixD& covMat )
{
   // build square-root C' of covariance matrix C (=> C = C'^2)
   Double_t sum = 0;
   for (Int_t i=0; i< m_size; i++) {
      sum = 0;
      for (Int_t j=0;j< i; j++) sum += (*m_sqrMat)(i,j) * (*m_sqrMat)(i,j);
      (*m_sqrMat)(i,i) = TMath::Sqrt(TMath::Abs(covMat(i,i) - sum));
      for (Int_t k=i+1 ;k<m_size; k++) {
         sum = 0;
         for (Int_t l=0; l<i; l++) sum += (*m_sqrMat)(k,l) * (*m_sqrMat)(i,l);
         (*m_sqrMat)(k,i) = (covMat(k,i) - sum) / (*m_sqrMat)(i,i);
      }
   }
}

Gfitter::GCorrGaussGen::GCorrGaussGen( const GCorrGaussGen& other )
   : GObject( other ),
     m_size ( other.m_size ),
     m_sqrMat( new TMatrixD(m_size,m_size) )
{
   // copy contructor
   InitClassName( "GCorrGaussGen" );

   for (Int_t i=0; i<m_size; i++) {
      for (Int_t j=0; j<m_size; j++) (*m_sqrMat)(i,j) = (*other.m_sqrMat)(i,j);
   }
}

Gfitter::GCorrGaussGen::GCorrGaussGen()
   : GObject(),
     m_size(0),
     m_sqrMat(0)
{   
   // default constructor
   InitClassName( "GCorrGaussGen" );
}

Gfitter::GCorrGaussGen::~GCorrGaussGen()
{
   // Destructor
   delete m_sqrMat;
}

void Gfitter::GCorrGaussGen::GetGaussRnd( TArrayD& v ) const
{
   // generate "m_size" correlated Gaussian random numbers

   // sanity check
   if (m_size > v.GetSize()) m_logger << kFATAL << "<GetGaussRnd> too short input vector" << GEndl   ;

   Double_t tmpVec[(const Int_t)m_size];

   for (Int_t i=0; i<m_size; i++) {
      Double_t x, y, z;
      y = gRandom->Rndm();
      z = gRandom->Rndm();
      x = 2*TMath::Pi()*z;
      tmpVec[i] = TMath::Sin(x) * TMath::Sqrt(-2.0*TMath::Log(y));
   }

   for (Int_t i=0; i<m_size; i++) {
      v[i] = 0;
      for (Int_t j=0; j<=i; j++) v[i] += (*m_sqrMat)(i,j) * tmpVec[j];
   }
}

Gfitter::GCorrGaussGen& Gfitter::GCorrGaussGen::operator=( const GCorrGaussGen& other )
{
   // assignment operator
   if (&other != this && other.m_size != m_size) {
      if (m_sqrMat) delete m_sqrMat;
      m_size = other.m_size;
      m_sqrMat = new TMatrixD( m_size, m_size );
   }
   if (&other != this) {
      for (Int_t i=0; i<m_size; i++){
         for (Int_t j=0; j<m_size; j++) (*m_sqrMat)(i,j) = (*other.m_sqrMat)(i,j);
      }
   }

   return *this;
}

