/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GDCardInterpreterBase                                                 *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>

#include "TFormula.h"

#include "Gfitter/GDCardInterpreterBase.h"
#include "Gfitter/GStringList.h"
#include "Gfitter/GUtils.h"
#include "Gfitter/GScalerLinear.h"
#include "Gfitter/GScalerFormula.h"

// NEVER change the following without the accord of Gfitter Group !
#define USE_SCALER_FORMULA__ kTRUE

using namespace std;

const Bool_t Gfitter::GDCardInterpreterBase::m_debug = kTRUE;

Gfitter::GDCardInterpreterBase::GDCardInterpreterBase( TString dataCardName )
   : m_dataCardName( dataCardName )
{
   InitClassName( "GDCardInterpreterBase" );
}

Gfitter::GDCardInterpreterBase::~GDCardInterpreterBase()
{}

Bool_t Gfitter::GDCardInterpreterBase::CheckList( const TList& list, Int_t minLength, TString where )
{
   if (list.GetSize() < minLength) {
      m_logger << kFATAL << "Wrong list range: " 
               << list.GetSize() << " < " <<  minLength << " in list: '" << where << "'" << GEndl;
      return kFALSE;
   }
   return kTRUE;
}

Bool_t Gfitter::GDCardInterpreterBase::CheckList( const GStringList& list, Int_t minLength, TString where )
{
   if (list.GetEntries() < minLength) {
      m_logger << kFATAL << "Wrong list range: " 
               << list.GetEntries() << " < " <<  minLength << " in list: '" << where << "'" << GEndl;
      return kFALSE;
   }
   return kTRUE;
}

Bool_t Gfitter::GDCardInterpreterBase::CheckList( const TList& list, const GStringList& reflist )
{
   // checks that "list" is contained in "reflist"   
   Bool_t retval = kTRUE;
   TIterator* it( list.MakeIterator() );
   TObject *obj = 0;
   while ((obj = it->Next())) {
      if (!reflist.Contains( obj->GetName() )) {
         m_logger << kWARNING << "Attribute: \"" << obj->GetName() << "\" not contained in reference list: " 
                  << "\"" << reflist.GetArgStr() << "\""
                  << GEndl;
         retval = kFALSE;
      }   
   }
   delete it;

   return retval;
}

// auxiliary methods
void Gfitter::GDCardInterpreterBase::InterpretScalers( const TString& sclstr, GSclPtrVec_t& scalerVec )
{
   // interpret scaling errors; options are:
   // C0: "+- 0.4 {beta, beta=0.5 +- 0.1} +- ..."
   // C1: "+0.4 -0.3 {beta, beta=0.5 +- 0.1} +- ..."
   
   // first decompose
   GStringList vlist( sclstr, "}" );
   if (vlist.GetEntries() < 1) {
      m_logger << kFATAL << "Mismatch in scaling part of \"Value\" tag: expected \"}\" : " << sclstr 
               << GEndl;
   }

   // loop over tokens, extract variable names and errors         
   for (Int_t i=0; i<vlist.GetEntries(); i++) {
      
      // first parse the parname
      TString parname = vlist[i];
      if (!parname.Contains("{")) {
         if (i != vlist.GetEntries()-1) {
            m_logger << kFATAL << "Mismatch in scaling part of \"Value\" tag: expected \"{\" : " 
                     << sclstr << " " << parname << GEndl ;
         }
         break;
      }
      parname.Remove( 0, parname.Index('{')+1 );
      
      GScalerBase* scaler = 0;

      if (USE_SCALER_FORMULA__) {
                  
         // now parse the errors
         TString parerr = vlist[i];
         parerr.Remove( parerr.Index('{') );            
         Double_t errp = 0, errm = 0;
         GUtils::InterpretErrorString( parerr, errp, errm, m_logger );                    

         // create the formula named by the parameter, which parses the expression
         // and create the scaler
         scaler = new GScalerFormula( parname, errp, errm );
      }
      else {

         // check sign in parameter name
         Int_t sign = 1;
         if (parname.Contains("-")) {
            sign = -1; 
            parname.ReplaceAll("-","");
         }
         else if (parname.Contains("+")) {
            sign = 1; 
            parname.ReplaceAll("+","");
         }
         
         // now parse the errors
         TString parerr = vlist[i];
         parerr.Remove( parerr.Index('{') );            
         Double_t errp = 0, errm = 0;
         GUtils::InterpretErrorString( parerr, errp, errm, m_logger );                    

         // and create the scaler
         scaler = new GScalerLinear( parname, errp, errm, sign );
      }

      // add GScaler class object to parameter
      scalerVec.push_back( scaler );
   }            
}

ostream& Gfitter::operator << ( ostream& os, const Gfitter::GDCardInterpreterBase& dcInterpreter ) 
{
	os << dcInterpreter.GetDCardName();
   return os;
}
