/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GData                                                                 *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <iostream>
#include <ctime>

#include "TString.h"
#include "TRandom.h"
#include "TF1.h"

#include "Gfitter/GData.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GMath.h"

ClassImp(Gfitter::GData)

Gfitter::GData::GData()
   : GDataBase(),
     m_value       ( 0 ),  
     m_errGaussp   ( 0 ),
     m_errGaussm   ( 0 ),
     m_errRangep   ( 0 ),
     m_errRangem   ( 0 ),
     m_errGaussSym ( 0 ),
     m_globalCorr  ( 0 )
{
   InitClassName( "GData" );   
}

Gfitter::GData::GData( Double_t value, Double_t errGauss )
   : GDataBase(),
     m_value       ( value),  
     m_errGaussp   ( errGauss ),
     m_errGaussm   ( errGauss ),
     m_errRangep   ( 0 ),
     m_errRangem   ( 0 ),
     m_errGaussSym ( errGauss ),
     m_globalCorr  ( 0 )
{
   InitClassName( "GData" );   
  
   m_fitValue = value;
}

Gfitter::GData::GData( Double_t value, 
                       Double_t errGaussp, Double_t errGaussm, Double_t errRangep, Double_t errRangem )
   : GDataBase(),
     m_value       ( value ),
     m_errGaussp   ( errGaussp ),
     m_errGaussm   ( errGaussm ),
     m_errRangep   ( errRangep ),
     m_errRangem   ( errRangem ),
     m_errGaussSym ( 0.5*(errGaussp + errGaussm) ),
     m_globalCorr  ( 0 )
{
   InitClassName( "GData" );   
   
   m_fitValue = value ;
}

Gfitter::GData::GData( const GInterval& interval )
   : GDataBase(),
     m_value       ( interval.GetMean() ) ,  // theory parameter: value is center of validity range
     m_errGaussp   ( 0 ),
     m_errGaussm   ( 0 ),
     m_errGaussSym ( 0 ),
     m_globalCorr  ( 0 )
{
   InitClassName( "GData" );   

   m_errRangep = m_errRangem = 0.5*interval.GetWidth();
   m_fitValue = interval.GetMean();
}

Gfitter::GData::GData( const GInterval& interval, Double_t initValue )
   : GDataBase(),
     m_value       ( interval.GetMean() ) ,  // theory parameter: value is center of validity range
     m_errGaussp   ( 0 ),
     m_errGaussm   ( 0 ),
     m_errGaussSym ( 0 ),
     m_globalCorr  ( 0 )
{
   InitClassName( "GData" );

   m_errRangep = m_errRangem = 0.5*interval.GetWidth();
   m_fitValue = initValue; //interval.GetMean();
}

Gfitter::GData::GData( const GData& other )
   : GDataBase( other ),
     m_value       ( other.m_value ),
     m_errGaussp   ( other.m_errGaussp ),
     m_errGaussm   ( other.m_errGaussm ),
     m_errRangep   ( other.m_errRangep ),
     m_errRangem   ( other.m_errRangem ),
     m_errGaussSym ( other.m_errGaussSym ),
     m_globalCorr  ( other.m_globalCorr )
{
   InitClassName( "GData" );   
}

Gfitter::GData::~GData()
{}

Double_t Gfitter::GData::GetGaussRnd(Double_t mean) const
{
   Double_t value = 0.;

   // random value for asymetric errors
   Double_t params[3];
   params[0] = mean;
   params[1] = m_errGaussp;
   params[2] = m_errGaussm;
   TF1 *f1 = new TF1( "asym Gauss", &GMath::AsymGauss, 
                      mean-(5.0*m_errGaussm), mean+(5.0*m_errGaussp), 3 );
   f1->SetParameters(params);
   value = f1->GetRandom();
   
   delete f1;
   return value;
}
 
Double_t Gfitter::GData::GetRndValue( Double_t /* mean */ )
{
   m_logger << kFATAL << "GetRndValue is not defined for GData" << GEndl;
   return 0.;
}

