/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GData                                                                 *
 *                                                                                *
 * Description:                                                                   *
 *      Container for parameter value                                             *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GData
#define Gfitter_GData

#include "Gfitter/GDataBase.h"

namespace Gfitter {

   class GInterval;
   class GParameter;

   class GData : public GDataBase {

   public:

      GData();
      GData( Double_t value, Double_t errGauss );
      GData( Double_t value, 
             Double_t errGaussp,     Double_t errGaussm, 
             Double_t errRangep = 0, Double_t errRangem = 0 );    
      GData( const GInterval& ); // constructor for theory parameters
      GData( const GInterval& interval, Double_t initValue ); // constructor for theory parameters with initial value
      GData( const GData& );     // copy constructor (cloning input GData)
      virtual ~GData();

      // accessors
      Double_t GetValue()     const { return m_value; }
      Bool_t   SetValue( Double_t v ); 

      Double_t GetErrGaussp()   const { return m_errGaussp; }
      Double_t GetErrGaussm()   const { return m_errGaussm; }
      Double_t GetErrGaussSym() const { return m_errGaussSym; } // note: do not use in fit !
      Double_t GetErrRangep()   const { return m_errRangep; }
      Double_t GetErrRangem()   const { return m_errRangem; }

      // global correlation coefficient
      Double_t GetGlobalCorr()  const { return m_globalCorr; }
      void     SetGlobalCorr( Double_t e ) { m_globalCorr = e; }      

      // total errors are defined by linear sum of Gaussian error and theory range
      Double_t GetErrTotp()     const { return m_errGaussp + m_errRangep; }
      Double_t GetErrTotm()     const { return m_errGaussm + m_errRangem; }
      Double_t GetErrTotSym()   const { return 0.5*(m_errGaussp + m_errRangep + m_errGaussm + m_errRangem); }

      // get random value around a certain mean value
      Double_t GetGaussRnd( Double_t mean ) const;

      // not used
      Double_t GetRndValue( Double_t mean );

   private:

      // a parameter is exceptoinally allowed to change private members
      friend class GParameter;

      Double_t       m_value;     // backup initial parameter setting
      Double_t       m_errGaussp;
      Double_t       m_errGaussm;
      Double_t       m_errRangep;
      Double_t       m_errRangem;
      Double_t       m_errGaussSym;
      Double_t       m_globalCorr;

      ClassDef(GData,1)
   };
}

//
// -------------------- i n l i n e   f u n c t i o n s ---------------------------------
//

inline Bool_t Gfitter::GData::SetValue( Double_t v ) 
{ 
   Bool_t signifChange = TMath::Abs((m_value-v)/m_value)>1E-10;
   m_value = v;

   return signifChange;
}

#endif
