/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GDataCurve                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <cmath>
#include <iostream>

#include "TROOT.h"
#include "TFile.h"
#include "TFormula.h"
#include "TF1.h"
#include "TMVA/TSpline1.h"
#include "TMVA/TSpline2.h"

#include "Gfitter/GDataCurve.h"
#include "Gfitter/GGraph2PDF.h"
#include "Gfitter/GMsgLogger.h"

using namespace std;

ClassImp(Gfitter::GDataCurve)

Gfitter::GDataCurve::GDataCurve()
   : GDataBase(),
     m_chi2hist( 0 ),
     m_graph   ( 0 ),
     m_offset  ( 0 )
{
   InitClassName( "GDataCurve" );   
}

Gfitter::GDataCurve::GDataCurve( TString fileName, TString histName, TString formula )
   : GDataBase()
{
   InitClassName( "GDataCurve" );   

   m_logger << kINFO << "Initializing a GDataCurve" << GEndl;
   m_logger << kINFO << "Root File: \"" << fileName 
            << "\" Histogram: \"" << histName << "\" Formula: \"" << formula << "\"" << GEndl;
   
   // get TFormula
   TFormula* tFormula= new TFormula( "converting_formula", formula.Data() );

   // open root file
   TFile *f1 = new TFile(fileName);
   if (f1==0) m_logger << kFATAL << "Error opening file" << fileName << GEndl;     

   // retrieve original histo
   TH1 *hist = (TH1F*)f1->Get(histName); 
   if (hist==0) {
      m_logger << kFATAL << "Error histogram \"" << histName << "\" in \"" << fileName << "\"" << GEndl;     
   }  
   gROOT->cd();

   // transform histo to chisq2 histo using formula
   m_chi2hist = (TH1*) hist->Clone(); 
   for (int ibin=0;ibin<hist->GetNbinsX(); ibin++) {
      m_chi2hist->SetBinContent(ibin, tFormula->Eval(hist->GetBinContent(ibin)));
   }
   
   // close file
   f1->Close();

   // get graph from histo
   m_graph = new TGraph( m_chi2hist );

   m_spline = new TMVA::TSpline1( "m_spline", m_graph );

   // initialisation
   m_fitValue  = m_chi2hist->GetBinCenter(m_chi2hist->GetMinimumBin()); 
}

Gfitter::GDataCurve::GDataCurve( const GDataCurve& dataCurve )
   : GDataBase( dataCurve ),
     m_chi2hist( 0 ),
     m_graph   ( 0 ),
     m_offset  ( 0 )
{   
   InitClassName( "GDataCurve" );   
   
   TString hname(dataCurve.m_chi2hist->GetName());
   m_chi2hist = (TH1*) dataCurve.GetHist().Clone("hname");
   m_graph    = new TGraph( dataCurve.GetGraph() );
   m_spline   = new TMVA::TSpline1( "m_spline", m_graph );
}

Gfitter::GDataCurve::~GDataCurve()
{
   if (m_graph)    delete m_graph;
   if (m_chi2hist) delete m_chi2hist;
   //if (m_spline)   delete m_spline;
}
 
Double_t Gfitter::GDataCurve::GetChiSquared( Double_t fitValue ) const
{
  Double_t retVal=0.;

  if (fitValue > this->GetMax() || m_fitValue < this->GetMin()) {
     retVal = 0;
  }
  else {

     retVal = m_chi2hist->Interpolate(fitValue) ;
     //retVal = m_spline->Eval(  fitValue - m_offset );

     // possible due to spline
     if (retVal < 0) {
        //m_logger << kWARNING << "<GetChiSquared> Chisq2= "<< retVal << GEndl;
        //retVal = 0.;
     }
     else if (isnan(retVal)) {
        m_logger << kWARNING << "<GetChiSquared> Chisq2 is \"nan\" at " << fitValue 
                 << " --> Setting to 1000. !! " << GEndl;
        
        retVal = 1000.;
     }
  }

  return retVal;
}

Double_t Gfitter::GDataCurve::GetMin() const
{
   return m_chi2hist->GetBinCenter(1);
}

Double_t Gfitter::GDataCurve::GetMax() const
{
   return m_chi2hist->GetBinCenter(m_chi2hist->GetNbinsX());
}

Double_t Gfitter::GDataCurve::GetValue() const
{
   return m_chi2hist->GetBinCenter(m_chi2hist->GetMinimumBin());
}

Bool_t Gfitter::GDataCurve::SetValue( Double_t ) 
{
   // move minimum of histogram here
   m_logger << kFATAL << "setting of values for GDataCurve still to be implemented" << GEndl;
   return 0;
}

Double_t Gfitter::GDataCurve::GetErrGaussp() const {
   // m_logger << kFATAL << "GetErrGaussp not yet implemented for GDataCurve" << GEndl;
   // MG: Set Error to 0
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrGaussm() const {
   //m_logger << kFATAL << "GetErrGaussm not yet implemented for GDataCurve" << GEndl;
   // MG: Set Error to 0
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrTotp() const {
   m_logger << kFATAL << "GetErrTotp not yet implemented for GDataCurve" << GEndl;
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrTotm() const {
   m_logger << kFATAL << "GetErrTotm not yet implemented for GDataCurve" << GEndl;
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrTotSym() const {
   m_logger << kFATAL << "GetErrTotSym not yet implemented for GDataCurve" << GEndl;
   return 0.;
}

Double_t Gfitter::GDataCurve::GetRndValue( Double_t mean )
{
   Double_t offset0 = m_chi2hist->GetBinCenter( m_chi2hist->GetMinimumBin() ) - mean;
   
   GGraph2PDF *pdf  = new GGraph2PDF(m_graph, offset0);
  
   TF1 *f1 = new TF1("pdf", pdf, &GGraph2PDF::Evaluate, this->GetMin(), this->GetMax(), 0, "GGraph2PDF", "Evaluate");
   m_logger << kINFO << "Here we are!!!" << GEndl;
   Double_t value = f1->GetRandom();
   
   m_offset =  offset0 + value - mean;
   m_logger << kINFO << offset0 << " + " <<  value << " - " << mean << " = " << m_offset << GEndl;
   delete f1;
   return value;
}

Double_t Gfitter::GDataCurve::GetGaussRnd( Double_t ) const
{
   m_logger << kFATAL << "GetGaussRnd is not defined for GDataCurve" << GEndl;
   return 0.;
}
