/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GFitterBase                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <iostream>
#include <iomanip>

#include "TString.h"
#include "TMath.h"
#include "TVectorD.h"

#include "Gfitter/GFitterBase.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GData.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GUtils.h"
#include "Gfitter/GInterval.h"

using namespace std;

Gfitter::GFitterBase::GFitterBase() 
   : m_chi2Min( 1e10 )
   , m_invErrorMatrix( 0 )
   , m_fitunstable(false)
   , m_chi2dummy(1000.)
{
   InitClassName( "GFitterBase" );
   GFitterBase::Initialise();   
}

Gfitter::GFitterBase::~GFitterBase()
{
   if (m_invErrorMatrix) delete m_invErrorMatrix;
}

void Gfitter::GFitterBase::Initialise()
{
   m_fitDuration = "unknown";
   m_fitPars.clear();
   UnsetFirstPassed();
}

void Gfitter::GFitterBase::checkErr( TString action, Int_t ierr, Int_t goodVal, Int_t fatalVal, Bool_t printChi2 )
{
   if (ierr == fatalVal) {
      if (!printChi2)
         m_logger << kFATAL << "<checkErr> Error return (" << ierr << ") for action: " << action << GEndl;
      else
         m_logger << kFATAL << "<checkErr> Error return (" << ierr << ") for action: " << action 
                  << " (chi2 = " << m_chi2Min << ")"
                  << GEndl;
   }
   else if (ierr != goodVal) {
      static Int_t printout_counter = 0;
      printout_counter++;
      if (printout_counter <= 10) {
         if (!printChi2)
            m_logger << kWARNING << "<checkErr> Warning (" << ierr << ") for action: " << action << GEndl;
         else
            m_logger << kWARNING << "<checkErr> Warning (" << ierr << ") for action: " << action 
                     << " (chi2 = " << m_chi2Min << ")"
                     << GEndl;
      }
      if (printout_counter == 10) {
         m_logger << kWARNING << "<checkErr> Suppress this printout from now on" << GEndl;
      }
   }     
}

void Gfitter::GFitterBase::FCN( Double_t &f, Double_t* fitPars, Int_t /* iflag */ )
{   
   // ------------------------------------------------------------------------------------
   // some initial debug output - only once in call
   if (IsFirstPass()) {
      const GParPtrVec_t& vu = gStore()->GetActiveUncorrelatedParameters();
      for (GParPtrVec_t::const_iterator par = vu.begin(); par != vu.end(); ++par) {
         if ((*par)->HasTheory()) {
            m_logger << kINFO << "Parameter with    theory enters uncorrelated chi2: " 
                     << "\"" << (*par)->GetFullName() << "\"" << GEndl;
         }
         else {
            m_logger << kINFO << "Parameter without theory enters uncorrelated chi2: " 
                     << "\"" << (*par)->GetFullName() << "\"" << GEndl;
         }
      }         
      const GParPtrVec_t& vc = gStore()->GetActiveCorrelatedParameters();
      for (GParPtrVec_t::const_iterator par = vc.begin(); par != vc.end(); ++par) {
         // MG: the following "if" is for a Scan of all parameters excl. their measurment
         //     because correlation matrix cannot be changed after the initialisation
         if( (*par)->IsActive() ){
            if ((*par)->HasTheory()) {
               m_logger << kINFO << "Parameter with    theory enters   correlated chi2: " 
                        << "\"" << (*par)->GetFullName() << "\"" << GEndl;
            }
            else {
               m_logger << kINFO << "Parameter without theory enters   correlated chi2: " 
                        << "\"" << (*par)->GetFullName() << "\"" << GEndl;
            }
         }
         else {
            m_logger << kINFO << "Parameter doesn't go in Fit, because it was set to false: "
               << "\"" << (*par)->GetFullName() << "\"" << GEndl;
         }
      }         
      SetFirstPassed();

      // reset minimum chi2
      m_chi2Min = 1e30;
      m_minval.resize(GetNumOfFitPars(),0);
   }

   // ------------------------------------------------------------------------------------

   // update GParameters in GStore
   Bool_t isnanpresent(kFALSE);

   for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) {
      // this sanity check is required due to the MINOS bug where a parameter can go out of its limits
      // and after that becomes NaN
      if (TMath::IsNaN(fitPars[ipar])) {
         isnanpresent = kTRUE;
         static Int_t printout_nan_counter = 0;
         printout_nan_counter++;
         //if (printout_nan_counter <= 5) {
           m_logger << kWARNING << "<FCN> parameter: " << GetFitPars()[ipar]->GetFullName() 
                    << " is NaN! Try to recover, old value " << GetFitPars()[ipar]->GetValue() << GEndl;
         //}
         //if (printout_nan_counter == 5) {
         //   m_logger << kWARNING << "<checkErr> Suppress this printout from now on" << GEndl;
         //}
         fitPars[ipar] = GetFitPars()[ipar]->GetValue();
      }
      GetFitPars()[ipar]->SetFitValue( fitPars[ipar] );
   }
   
   GParPtrVec_t::const_iterator par;
   const GParPtrVec_t& v = gStore()->GetActiveParameters();
   
   // debug output
   if ((m_logger.GetMinLevel() <= kDEBUG) || isnanpresent) {
      for (par = v.begin(); par != v.end(); par++) {
         Double_t value;
         if (!(*par)->HasTheory()) value = (*par)->GetFitValue();
         else                      value = (*par)->GetTheoryPrediction();
         m_logger << kDEBUG << "Fit par " << (*par)->GetFullName()
                  << "\" initial value (v/fit): ("
                  << (*par)->GetValue() << "/" << value << ")"
                  << ", fit range: [" << (*par)->GetFitRange().GetMin() << ", "
                  << (*par)->GetFitRange().GetMax() << "]"
                  << ", fit step: " << (*par)->GetFitStep()
                  << ", chi2: " << (*par)->GetChiSquared()
                  << GEndl;
         if (TMath::IsNaN( value )) m_logger << kFATAL << "NAN !!!" << GEndl;   
      }
   }
	   
   // compute chi-squared
   Double_t chi2 = 0;
   
   // uncorrelated parameters
   const GParPtrVec_t& vu = gStore()->GetActiveUncorrelatedParameters();
   for (par = vu.begin(); par != vu.end(); ++par) {
     chi2 += (*par)->GetChiSquared();
   }

   // correlated pars
   if (gStore()->IsCorrelationActive()) {
      const GParPtrVec_t& vc = gStore()->GetActiveCorrelatedParameters();
      const Int_t npar = vc.size();
      Double_t delta[npar];
      for (Int_t ipar = 0; ipar < npar; ++ipar) {
         // MG: check if parameter is active, important for Scan of all parameters excl. their measurment
         //     Note: Correlation matrix cannot be changed after the initialisation !
         if( vc[ipar]->IsActive() ) {
            delta[ipar] = vc[ipar]->GetDeviation();
            chi2       += delta[ipar] * (*m_invErrorMatrix)(ipar,ipar) * delta[ipar];
         }
      }
      for (Int_t ipar = 0; ipar < npar; ++ipar) {
         for (Int_t jpar = ipar+1; jpar < npar; jpar++) {
            // MG: check if parameter is active, important for Scan of all parameters excl. their measurment
            //     Note: Correlation matrix cannot be changed after the initialisation !
            if( vc[ipar]->IsActive() && vc[jpar]->IsActive() ) { 
               chi2 += 2 * delta[ipar] * (*m_invErrorMatrix)(ipar,jpar) * delta[jpar];
            }
         }
      }
   }

   // add penalties
   double chi2nopen = chi2;
   const GParPtrVec_t& pe = GetPenalisedPars();
   for (par = pe.begin(); par != pe.end(); ++par) chi2 += (*par)->GetPenaltyChiSquared();
   f = chi2; 

   // Dump chi2 on the screen
   m_logger << kDEBUG << "Combined chi2 for this point: " << chi2 << ". W/o penalty: " << chi2nopen <<  GEndl;

   // save minimum chi2 result and parameters
   if (f < m_chi2Min) { 
     m_chi2Min = f;
     for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) {
       Double_t parvalue = GetFitPars()[ipar]->GetFitValue();
       m_minval[ipar] = parvalue;
       if (TMath::IsNaN( parvalue )) m_logger << kFATAL << "NAN !!!" << GEndl;    
     }
   }

   // MB : in case of unstable (susy) fit, possibility to 'stabalize' fit with dummy chi2 value
   // Normally not/never needed
   if ( this->GetFitUnstable() ) { f=m_chi2dummy; }
}

void Gfitter::GFitterBase::InitErrorMatrix()
{
   if (gStore()->IsCorrelationActive()) {
      // first print correlations matrix
      m_logger << kINFO << "Correlation matrix: " << GEndl;
      GUtils::FormattedOutput( *gStore()->GetCorrelationMatrix( gStore()->GetActiveCorrelatedParameters() ), 
                               gStore()->GetActiveCorrelatedParameters(), m_logger, "%1.3g" );
      
      TMatrixD* covMatrix = gStore()->GetCovarianceMatrix( gStore()->GetActiveCorrelatedParameters() );
      m_logger << kINFO << "Covariance matrix: " << GEndl;
      GUtils::FormattedOutput( *covMatrix, gStore()->GetActiveCorrelatedParameters(), m_logger, "%1.2g" );
      
      if (covMatrix != 0) {

         // sanity check: covariance matrix must be strictly positiv definit 
         // (x^T C > 0 <==> det(C) > 0, but also all eigenvalues > 0)
         TVectorD eigenv( covMatrix->GetNrows() );
         covMatrix->EigenVectors( eigenv );
         for (Int_t i=0; i<covMatrix->GetNrows(); ++i) {
            if (eigenv(i) <= 0) {
               m_logger << kFATAL 
                        << "The covariance matrix has eigenvalues <= 0: (EV[" << i << "] = " << eigenv(i) << ")."
                        << "Therefore, the matrix is not strictly positif definit and cannot be used to solve "
                        << "a least square problem. Please check the input correlations."
                        << GEndl;
            }
         }

         covMatrix->Invert();
         if (m_invErrorMatrix) { delete m_invErrorMatrix; m_invErrorMatrix = 0; }
         m_invErrorMatrix = new TMatrixD( *covMatrix ); 
         m_logger << kINFO << "Inverse covariance matrix: " << GEndl;
         GUtils::FormattedOutput( *m_invErrorMatrix, gStore()->GetActiveCorrelatedParameters(), m_logger, "%1.2g" );
      }
   }
}

void Gfitter::GFitterBase::PrintConfiguration()
{
   m_logger << kINFO << "Number of active parameters in FCN method: " 
            << gStore()->GetActiveParameters().size() 
            << " (correlated: " << gStore()->GetActiveCorrelatedParameters().size() 
            << ", uncorrelated: " << gStore()->GetActiveUncorrelatedParameters().size() << ")" << GEndl;
   
   const GParPtrVec_t& v = gStore()->GetActiveParameters();
   GParPtrVec_t::const_iterator par;
   Int_t ipar = 0;
   for (par = v.begin(); par != v.end(); par++) {
      m_logger << kINFO << "Active parameter(" << ipar++ << ") : \"" 
               << (*par)->GetFullName() << "\"" << GEndl;
   }

   // fit parameters
   m_logger << kINFO << "Number of fit parameters: " << GetFitPars().size() << GEndl;
   const GParPtrVec_t& v2 = GetFitPars();
   ipar = 0;
   for (par = v2.begin(); par != v2.end(); par++) {
      m_logger << kINFO << "Fit par [" << ipar++ << "]: \"" << setw(8) << (*par)->GetParName() 
               << "\" initial value (v/fit): (" 
               << (*par)->GetValue() << "/" << (*par)->GetFitValue() << ")"
               << ", fit range: [" << (*par)->GetFitRange().GetMin() << ", " 
               << (*par)->GetFitRange().GetMax() << "]" 
               << ", fit step: " << (*par)->GetFitStep()
               << GEndl;
   }

   // penalised parameters
   m_logger << kINFO << "Number of penalised parameters: " << GetPenalisedPars().size() << GEndl;
   const GParPtrVec_t& v3 = GetPenalisedPars();
   ipar = 0;
   for (par = v3.begin(); par != v3.end(); par++) {
      m_logger << kINFO << "Pen-par [" << ipar++ << "]: \"" << setw(8) << (*par)->GetParName() 
               << "\" initial value (v/fit): (" 
               << (*par)->GetValue() << "/" << (*par)->GetFitValue() << ")"
               << ", fit range: [" << (*par)->GetFitRange().GetMin() << ", " 
               << (*par)->GetFitRange().GetMax() << "]" << GEndl;
   }
}

