/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GFitterBase                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Abstract class for fitter                                                 *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GFitterBase
#define Gfitter_GFitterBase

#include "TMatrixD.h"

#include "Gfitter/GObject.h"
#include "Gfitter/GData.h"
#include "Gfitter/GStore.h"

namespace Gfitter {

   class GStore;
      
   class GFitterBase : public GObject {
      
   public:
      
      GFitterBase(); // default constructor
      virtual ~GFitterBase();

      // initialise fitter
      virtual void     Initialise();

      // perform fit, and return minimum chi-squared
      virtual Double_t ExecuteFit() = 0;            
      virtual Double_t GetCurrentChi2() = 0;

      //scan two variables using MNCONTour
      //      virtual void DrawContour(int npoints, Double_t dchi2) =0;
      virtual void DrawContour( GParameter* gpar1, GParameter* gpar2, 
                                Int_t npoints, Double_t dchi2, Double_t* npx, Double_t* npy ) = 0;

      // access to fit parameters
      Int_t               GetNumOfFitPars() const { return m_npars; }
      void                SetNumOfFitPars( Int_t n ) { m_npars = n; }
      const GParPtrVec_t& GetFitPars()       const { return m_fitPars; }
      GParPtrVec_t&       GetFitPars()             { return m_fitPars; }

      // penalized parameters
      const GParPtrVec_t& GetPenalisedPars() const { return m_penalisedPars; }
      GParPtrVec_t&       GetPenalisedPars()       { return m_penalisedPars; }

      // access to estimator and EDM after fit convergence
      Double_t            GetEstimator() const { return m_estimator; }
      void                SetEstimator( Double_t e ) { m_estimator = e; }
      Double_t            GetEDM() const { return m_edm; }
      void                SetEDM( Double_t e ) { m_edm = e; }

      inline void 	  SetFitUnstable(const Bool_t& fitunstable = kTRUE) { m_fitunstable=fitunstable; }
      inline Bool_t       GetFitUnstable() { return m_fitunstable; }

      // print info
      void PrintConfiguration();

      // update the fit results of all parameters
      virtual void UpdateResults() = 0;

      // fint timing
      const TString GetFitDuration() const { return m_fitDuration; }

   protected:
      

      // method that computes the minimized estimator
      void FCN( Double_t &f, Double_t* fitPars, Int_t iflag );
      
      GParPtrVec_t        m_fitPars;             // the fit parameters      
      Int_t               m_npars;               // number of fit parameters (free + fixed)
      Double_t            m_estimator;           // value of estimator after fit convergence
      Double_t            m_edm;                 // value of EDM after fit convergence
      Double_t            m_chi2Min;             // minimum chi2

      std::vector<Double_t> m_minval;          // fitpars corresponding to minchi2

      // flag for debug printout
      Bool_t              IsFirstPass() const { return m_firstPass; }
      void                UnsetFirstPassed()  { m_firstPass = kTRUE; }
      void                SetFirstPassed()    { m_firstPass = kFALSE; }
      Bool_t              m_firstPass;           // output if first pass

      // error matrix and its inverse
      void InitErrorMatrix();
      TMatrixD*           m_invErrorMatrix;

      // MB hack: boolian in case of fit instability
      Bool_t 		  m_fitunstable;
      Double_t            m_chi2dummy;

      // list of penalties
      GParPtrVec_t        m_penalisedPars;       // contains all parameters that need to be penalised

      // verification
      void checkErr( TString action, Int_t ierr, Int_t goodVal = 0, Int_t fatalVal = -123, 
                     Bool_t printChi2 = kFALSE );

      // fit timing
      TString             m_fitDuration;
   };   

}

#endif
