/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GFunctionOfFreeParameter                                              *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>
#include "TAxis.h"
#include "TH1F.h"
#include "TH2F.h"
#include "TTree.h"
#include "TMath.h"
#include "TGraph.h"
#include "Gfitter/GFunctionOfFreeParameter.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTimer.h"

using namespace std;
using namespace Gfitter;

Gfitter::GFunctionOfFreeParameter::GFunctionOfFreeParameter()
{
   InitClassName( "GFunctionOfFreeParameter" );
}

Gfitter::GFunctionOfFreeParameter::~GFunctionOfFreeParameter()
{}

void Gfitter::GFunctionOfFreeParameter::FctOfFreePara( const TString& par, Int_t nbins )
{
   GParameter* gpar    = gStore()->GetParameter( par );
   if (gpar == 0) m_logger << kFATAL << "<FctOfFreePara> Fatal error: unknown parameter \"" << par << "\": "
                           << " check datacard" << GEndl;

   FctOfFreePara( gpar, nbins );
}

void Gfitter::GFunctionOfFreeParameter::FctOfFreePara( GParameter* gpar, Int_t nbins)
{
   // sanity check
   if (gpar == 0) m_logger << kFATAL << "<FctOfFreePara> Zero GParameter pointer" << GEndl;

   if (gpar->HasTheory()) m_logger << kFATAL << "<FctOfFreePara> Only free floating parameters can be scanned!" << GEndl;

   m_logger << kINFO << "Scanning parameter \"" << gpar->GetParName() << "\""
            << " (full name: \"" << gpar->GetFullName() << "\")"
            << " in interval [" << gpar->GetScanRange().GetMin() << ", " << gpar->GetScanRange().GetMax() << "]"
            << GEndl;
   m_logger << kINFO << "Number of steps in scan: " << nbins << GEndl;

      
   // book parameter tree
   TTree* Tree = new TTree( Form( "results_%s", gpar->GetParNameWONSpace().Data() ), 
                            Form( "Results for parameter %s", gpar->GetParNameWONSpace().Data() ) );
   
   // how many active parameters with theory
   Int_t num = gStore()->GetNumOfParameters() - gStore()->GetNumOfActiveParametersWithoutTheory();
   
   // create the tree branches
   Double_t x_val[num];
   Int_t i = 0;

   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); par1++) {
      if( (*par1)->HasTheory() ){
         Tree->Branch( (*par1)->GetFullNameWONSpace() , &x_val[i], "x_val/D" );
         i++;
      }
   }

   Double_t x_scanval;
   Tree->Branch( gpar->GetFullNameWONSpace() , &x_scanval, "x_scanval/D" );

   // backup paramter settings
   gpar->Backup();
  
   // clarify for later that this parameter is going to be scanned
   //gpar->SetScanned( kTRUE );      
   
   // compute step size
   Double_t min  = gpar->GetScanRange().GetMin();
   Double_t max  = gpar->GetScanRange().GetMax();
   Double_t step = (max - min) / (nbins-1);
   
   // scan free floating parameter and compute theory predictions
   m_logger << kINFO << "Scanning ... please be patient" << GEndl;        
   for (Int_t ibin = 0; ibin < nbins; ibin++) {
 
      gpar->SetFitValue( ibin*step + min  );
            
      x_scanval = gpar->GetFitValue();
      	  
      // fill the Tree 
      Int_t k=0;
      for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
           par1 != gStore()->GetActiveParameters().end(); par1++) {
         if   ( (*par1)->HasTheory() ) { 
            x_val[k]  = (*par1)->GetTheoryPrediction();
            k++;
         }
      }
      Tree->Fill();
   }

   m_logger << kINFO << "The results of all Theory-Parameters are fillesd in a tree!!!" 
            << "                                      " << GEndl;    
   
   // recover initial state
   gpar->Recover();
   
   Tree->Write();
   delete Tree;
   
}

ostream& Gfitter::operator << ( ostream& os, const Gfitter::GFunctionOfFreeParameter& /* scan */ ) 
{
   os << 0;
   return os;
}
