/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GGSLMinimizer                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "TFitter.h"
#include "Gfitter/GGSLMinimizer.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GData.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GResultEval.h"
#include "Gfitter/GTimer.h"
#include "Gfitter/GMinimizerFcn.h"

#include "Math/Minimizer.h"
#include "TMatrixDSym.h"
#include "TObjArray.h"
#include "TObjString.h"
#include "TMath.h"
#include "TROOT.h"

using namespace std;

ClassImp(Gfitter::GGSLMinimizer)

Gfitter::GGSLMinimizer::GGSLMinimizer() 
   : GFitterBase(),
     _theFitter( 0 ),
     m_useImprove( kFALSE ),
     m_useMinos( kFALSE ),
     m_isFirstInit( kTRUE ),
     m_minimizerType("Minuit2"),
     m_minimizerAlg("migrad"),
     m_eps(1.0),
     m_fcn(0),
     m_doPrefitScan(true),
     m_doPostfitScan(false),
     m_doCorner(false),
     m_doRandomize(false),
     m_nfit(1),
     m_chi2minLoop(1E20),
     m_edmLoop(1E20),
     m_deactivated(false),
     m_printSummary(false)
{
   InitClassName( "GGSLMinimizer" );
}

Gfitter::GGSLMinimizer::GGSLMinimizer(const char* minimizerType, const char* minimizerAlg)
   : GFitterBase(),
     _theFitter( 0 ),
     m_useImprove( kFALSE ),
     m_useMinos( kFALSE ),
     m_isFirstInit( kTRUE ),
     m_minimizerType(minimizerType),
     m_minimizerAlg(minimizerAlg),
     m_eps(1.0),
     m_fcn(0),
     m_doPrefitScan(true),
     m_doPostfitScan(false),
     m_doCorner(false),
     m_doRandomize(false),
     m_nfit(1),
     m_chi2minLoop(1E20),
     m_edmLoop(1E20),
     m_deactivated(false)
{
   InitClassName( "GGSLMinimizer" );
}

Gfitter::GGSLMinimizer::~GGSLMinimizer()
{
   if (_theFitter)  delete _theFitter;
   if (m_fcn)       delete m_fcn;
}

void Gfitter::GGSLMinimizer::Initialise()
{
   // default: (susy) fit works fine of course
   this->SetFitUnstable(false);

   // run the prefitter first
   if (gStore()->GetPreFitter())
     runPreFitter();

   // default: (susy) fit works fine of course
   this->SetFitUnstable(false);

   // determine name of fitter class
   m_myname = GetName();
   TObjArray* fitNameArr = m_myname.Tokenize("::");
   TObjString* objstring = (TObjString*)fitNameArr->At( fitNameArr->GetEntries()-1 );
   m_myname = objstring->GetString(); 
   delete fitNameArr;

   // Parameters to deactivate for this fit?
   const TString& deactivateStr = ( gStore()->ExistVariable( m_myname+"::Deactivate" ) ?
                                    gStore()->GetVariable( m_myname+"::Deactivate" )->GetStringValue() : "" );
   fitNameArr = deactivateStr.Tokenize(":");
   m_deactive.clear();
   for (Int_t ipar=0; ipar<fitNameArr->GetEntries(); ++ipar) {
     TString parName = ((TObjString*)fitNameArr->At(ipar))->GetString();
     GParameter* par = gStore()->GetParameter( parName );
     if (par) {
       m_deactive.push_back( par );
     }
   }
   delete fitNameArr;
   if (!m_deactivated) deactivate(); // reactivate after execute() or contour()

   // Print summary line after fit?
   m_printSummary = ( gStore()->ExistVariable( m_myname+"::PrintSummary" ) ?
                      gStore()->GetVariable( m_myname+"::PrintSummary" )->GetBoolValue() : false );

   m_This = this;      

   // base class call
   GFitterBase::Initialise();
   
   if( !m_isFirstInit )
      GetThisPtr()->SetFirstPassed();
   
   // delete Fitter instance if existing
   if (_theFitter) { delete _theFitter; } _theFitter=0;   
   if (m_fcn)      { delete m_fcn; }      m_fcn=0;

   // reset vectors
   GetFitPars().clear();
   GetPenalisedPars().clear();      
   
   // instantiate minuit
   _theFitter = new ROOT::Fit::Fitter;
   m_fcn = new GMinimizerFcn(this,gStore()->GetNumOfActiveParametersWithoutTheory());
   _theFitter->Config().SetMinimizer(m_minimizerType.c_str(),m_minimizerAlg.c_str());

   // Set Minuit tolerance. Default=1.0
   m_eps = (gStore()->ExistVariable( m_myname+"::Eps" ) ? 
           gStore()->GetVariable( m_myname+"::Eps" )->GetFloatValue() : 1.0);
   _theFitter->Config().MinimizerOptions().SetTolerance(m_eps);

   // default max number of calls
   _theFitter->Config().MinimizerOptions().SetMaxIterations(10000*gStore()->GetNumOfActiveParametersWithoutTheory());
   _theFitter->Config().MinimizerOptions().SetMaxFunctionCalls(10000*gStore()->GetNumOfActiveParametersWithoutTheory());

   
   // minuit-specific settings
   Double_t args[10];

   args[0] = (gStore()->ExistVariable( m_myname+"::PrintLevel" ) ? 
              gStore()->GetVariable( m_myname+"::PrintLevel" )->GetIntValue() : -1);
   Int_t printLevel = static_cast<Int_t>(args[0]);
   _theFitter->Config().MinimizerOptions().SetPrintLevel(printLevel+1) ;

   // define fit parameters 
   std::vector<ROOT::Fit::ParameterSettings>& parameters = _theFitter->Config().ParamsSettings();
   Int_t npars = 0;
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
   for (; par != gStore()->GetActiveParameters().end(); ++par) {
      
      // is theoretical parameter (as opposed to measurement)
      // ... and is not actually a parameter that is scanned
      // (if it were scanned, it should enter the fit, but not be free to vary)
         
      // NOTE: it may occur that several measurements of a same GParameter "A"
      // were added to the datacard. If that parameter does not have a theory, 
      // we should add only once the parameter to the fit, and the corresponding 
      // A::m_fitValues need all to be set to this single free parameter
      
      // We implement this by adding a flag "IsFirst()" to each parameter. If they 
      // are unique, the flag is "T". If they are non-unique, the flag is only "T"
      // for the first parameter added to the GStore. The parameter will then hold 
      // a list of pointer to the other parameters of the same name, and update their
      // fit values  

      if ((*par)->VariesInFit()) {

         m_logger << kDEBUG << "Now adding parameter to fit configuration : " << (*par)->GetParName() << GEndl;

         // save name and pointer to parameter that is free in the fit
         GetFitPars().push_back( *par );
         
         // sanity check
         if ((*par)->GetFitStep() <= 0) m_logger << kFATAL << "Huge troubles with fit step of par: " 
                                                 << (*par)->GetParName() << GEndl;
    
         // book the fit parameter
         npars++;
         parameters.push_back(ROOT::Fit::ParameterSettings((*par)->GetParName().Data(),
                                                           (*par)->GetFitValue(),
                                                           (*par)->GetFitStep(),
                                                           (*par)->GetFitRange().GetMin(),
							   (*par)->GetFitRange().GetMax() ));
      }  
   }

   // if the parameter has a theory and should be scanned, we need to add 
   // a penalty function to the fit that guarantees that the theory always
   // corresponds to the wanted value in the scan   

   // JH: loop over all Parameters needed here! Scan!
   for (par = gStore()->GetParameters().begin(); par != gStore()->GetParameters().end(); ++par) {
      // change from MG: We do not need IsFirst(), don't we?
      if ((*par)->HasTheory() && (*par)->IsScanned() ) { 
         m_logger << kINFO << "Setting penalised Parameter: " << (*par)->GetParName() << GEndl;
         GetPenalisedPars().push_back( (*par) );
      }
   }        
   
   // set internally the number of fit parameters (needed due to TFitter "feature" of 
   // "current" free parameters given to FCN only)
   this->SetNumOfFitPars( npars );

   m_bestvalsLoop.clear();
   m_bestvalsLoop.resize(GetNumOfFitPars(),0);
   
   Int_t errorLevel = (gStore()->ExistVariable( m_myname+"::ErrorLevel" ) ?
                       gStore()->GetVariable( m_myname+"::ErrorLevel" )->GetIntValue() : 1);
   _theFitter->Config().MinimizerOptions().SetErrorDef(errorLevel);

   // do prefit scan?
   if (gStore()->ExistVariable( m_myname+"::DoPrefitScan" )) {
      m_doPrefitScan = gStore()->GetVariable( m_myname+"::DoPrefitScan" )->GetBoolValue();
   }

   // do postfit scan?
   if (gStore()->ExistVariable( m_myname+"::DoPostfitScan" )) {
      m_doPostfitScan = gStore()->GetVariable( m_myname+"::DoPostfitScan" )->GetBoolValue();
   }

   // use improve and/or minos ?
   if (gStore()->ExistVariable( m_myname+"::UseImprove" )) {
      m_useImprove = gStore()->GetVariable( m_myname+"::UseImprove" )->GetBoolValue();
   }
   if (gStore()->ExistVariable( m_myname+"::UseMinos" )) {
      m_useMinos = gStore()->GetVariable( m_myname+"::UseMinos" )->GetBoolValue();
   }

   // print warnings ?
   //if (gStore()->ExistVariable( m_myname+"::NoWarnings" )) {
   //   if (gStore()->GetVariable( m_myname+"::NoWarnings" )->GetBoolValue())
   //      this->checkErr( "SET NOWARNINGS", m_fitter->ExecuteCommand( "SET NOWARNINGS", args, 0 ) );
   //}
   // equivalent for _theFitter ?

   // define fit strategy
   if (gStore()->ExistVariable( m_myname+"::Strategy" )) {
      args[0] = gStore()->GetVariable( m_myname+"::Strategy" )->GetIntValue();
   }
   else args[0] = 2;
   Int_t istrat = Int_t(args[0]);
   _theFitter->Config().MinimizerOptions().SetStrategy(istrat) ;   

   // sanity check before computing correlation errors 
   if (!gStore()->IsCorrelationActive() && 
       gStore()->GetActiveParameters().size() != gStore()->GetActiveUncorrelatedParameters().size()) {
      m_logger << kFATAL << "<Initialise> (Correlations inactive) error in number of parameters: "
               << gStore()->GetActiveParameters().size() << " | " 
               << gStore()->GetActiveCorrelatedParameters().size() << " | " 
               << gStore()->GetActiveUncorrelatedParameters().size() << GEndl;
   }
   
   // init covariance matrices and/or diagonal errors
   if (gStore()->IsCorrelationActive()) this->InitErrorMatrix();

   // some debug output, only when function is called first
   if( m_isFirstInit ){
      PrintConfiguration();
      m_isFirstInit = kFALSE;
   }

}
   
Double_t Gfitter::GGSLMinimizer::GetCurrentChi2()
{
   // update fit parameters
   Double_t chi2, *gin( 0 ), *fitPars( 0 );
   Int_t    iflag( 0 ), npars( 0 );

   fitPars = new Double_t[GetNumOfFitPars()];
   for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) {
     fitPars[ipar] = GetFitPars()[ipar]->GetFitValue();
   }

   this->IFCN( npars, gin, chi2, fitPars, iflag );

   delete [] fitPars; 
   return chi2;
}

Double_t Gfitter::GGSLMinimizer::ExecuteFit()
{   
  // Parameters to deactivate for this fit?
  if (!m_deactivated) deactivate();

  // short cut if no free parameters: compute FCN right away
  if (GetFitPars().size() == 0) {
     Double_t chi2, *gin( 0 ), *fitPars( 0 );
     Int_t    iflag( 0 ), npars( 0 );
     this->IFCN( npars, gin, chi2, fitPars, iflag );
     return chi2;
  }   

  // init timer
  GTimer timer;

  // start fit loop
  // default: m_nfit = 1
  for (Int_t ifit=0; ifit<m_nfit; ++ifit) 
  {  
    Double_t _chi2(1e20), _edm(0);

    // apply pre-fit scan ?
    if (m_doPrefitScan) { (void) runPreFitScan(); }

    // random initial fit values
    if (m_doRandomize || m_doCorner) { 
      m_logger << kINFO << "Now at loop # " << ifit+1 << "/" << m_nfit << GEndl;
      setInitValues();
    }

    // reset minimum chi2 --> this parameter is to check that the fitter really returns the 
    //                        minimum chi2 found
    m_chi2Min = 1e20;

    // do the fit!
    doFit();

    // improved follow-up minimization, eg. migradimproved or MINOS
    if (m_useImprove || m_useMinos) followUpFits();

    // sanity check
    if (GetFitPars().size() != (UInt_t)_theFitter->Result().Parameters().size()) {
       m_logger << kFATAL << "<ExecuteFit> Mismatch in number of parameters: "
                << GetFitPars().size() << " != " << _theFitter->Result().Parameters().size() << GEndl;
    }

    // retrieve fit result
    _chi2 = _theFitter->Result().MinFcnValue() ; 
    _edm  = _theFitter->Result().Edm() ;

    // store best fit loop results
    if (_chi2<m_chi2minLoop) {
      m_chi2minLoop = _chi2;
      m_edmLoop = _edm;
      for (UInt_t ipar=0; ipar<GetFitPars().size(); ++ipar)
        m_bestvalsLoop[ipar] = _theFitter->Result().Parameters()[ipar] ;
    }

    // do postfit parameter scan - store in m_bestvalsLoop
    //if (m_doPostfitScan) { (void) runPostFitScan(); }

    // has the minimum chi2 during the fit actually been returned ? If not, overwrite.
    static Int_t chi2_message_counter = 0;
    if (m_chi2Min < m_chi2minLoop /*+ 100*_edm*/) {
       chi2_message_counter++;
       if (chi2_message_counter <= 10) {
          m_logger << kWARNING << "Fit did not return solution corresponding to minimum chi2. Fit returned: chi2 = "
                   << _chi2 << "; minimum was: chi2 = " << m_chi2Min 
                   << " (EDM = " << _edm << ")"
                   << " ==> correct chi2"
                   << GEndl;
       }
       if (chi2_message_counter == 10) {
          m_logger << kWARNING << "Suppress this message from now on" << GEndl;
       }
       m_chi2minLoop = m_chi2Min;
       for (UInt_t ipar=0; ipar<GetFitPars().size(); ++ipar)
         m_bestvalsLoop[ipar] = m_minval[ipar] ;
    }

    m_logger << kDEBUG << "Latest chi2 minimum = " << m_chi2minLoop << " vs " << m_chi2Min << GEndl;

  } // end of fit loop   

  // measure time for fit
  m_fitDuration = timer.GetElapsedTime();
  m_logger << kINFO << "Elapsed time: " << timer.GetElapsedTime() << GEndl;

  // store parameters of fit result
  this->SetEstimator( m_chi2minLoop );
  this->SetEDM( m_edmLoop );      
  m_logger << kINFO << "Best chi2 = " << m_chi2minLoop 
                    << " , with edm = " << m_edmLoop << GEndl;

  std::vector<ROOT::Fit::ParameterSettings>& parameters = _theFitter->Config().ParamsSettings();
  for (UInt_t ipar=0; ipar<GetFitPars().size(); ++ipar) {
     GetFitPars()[ipar]->SetFitValue( m_bestvalsLoop[ipar] );
     parameters[ipar].SetValue( m_bestvalsLoop[ipar] );
     m_logger << kINFO << ">> With parameter : " << GetFitPars()[ipar]->GetParName() 
                       << " = " << m_bestvalsLoop[ipar] << GEndl;
  }

  // print fit summary, easily greppable
  int randomSeed(0);
  if (gStore()->ExistVariable( "GfitterFlags::RandomSeed" ))
    randomSeed = gStore()->GetVariable( "GfitterFlags::RandomSeed" )->GetIntValue();
  const TString summaryprefix = Form("Fit summary of randomSeed=%d : ", randomSeed);
  double curchi2 = this->GetCurrentChi2();
  m_logger << kINFO << summaryprefix << "Current chi2 = " << curchi2 << GEndl;

  GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
  for (; par != gStore()->GetActiveParameters().end(); ++par) {
     Double_t value;
     if (!(*par)->HasTheory()) value = (*par)->GetFitValue();
     else                      value = (*par)->GetTheoryPrediction();
     m_logger << kDEBUG 
              << summaryprefix
              << "Fit par " << (*par)->GetFullName()
              << "\" initial value (v/fit): ("
              << (*par)->GetValue() << "/" << value << ")"
              << ", fit range: [" << (*par)->GetFitRange().GetMin() << ", "
              << (*par)->GetFitRange().GetMax() << "]"
              << ", fit step: " << (*par)->GetFitStep()
              << ", chi2: " << (*par)->GetChiSquared()
              << GEndl;
     if ( TMath::IsNaN(value) ) m_logger << kFATAL << "NAN !!!" << GEndl;
  }

  // one line summary of result, useful for grepping output
  if (m_printSummary) {
    TString parname;
    Double_t val, init, errp, errm, errsym, globcor, fitstep, pchi2;
    int status   = _theFitter->Result().Status() ; 
    int covqual  = _theFitter->GetMinimizer()->CovMatrixStatus() ;

    m_logger << kINFO << "Begin of " << summaryprefix << GEndl; // 'empty' line in summary
    m_logger << kINFO << summaryprefix << "GREPME : "; 
    m_logger << "chi2min=" << curchi2 << " chi2loop=" << m_chi2minLoop << " edm=" << m_edmLoop << " status=" << status << " covqual=" << covqual;

    // fit parameters
    for (UInt_t ipar=0; ipar<GetFitPars().size(); ipar++) {
      parname = GetFitPars()[ipar]->GetFullName();  //GetParName();
      parname.ReplaceAll(":","_");
      parname.ReplaceAll(".","_");
      val     = m_bestvalsLoop[ipar]; //_theFitter->Result().Parameters()[ipar] ;
      init    = GetFitPars()[ipar]->GetValue() ;
      errsym  = _theFitter->Result().Error(ipar) ;
      errp    = _theFitter->Result().UpperError(ipar) ;
      errm    = _theFitter->Result().LowerError(ipar) ;
      globcor = _theFitter->Result().GlobalCC(ipar) ;
      fitstep = GetFitPars()[ipar]->GetFitStep() ;
      pchi2   = GetFitPars()[ipar]->GetChiSquared() ;

      m_logger << " " << parname << "=" << val 
               << " " << parname << "_init=" << init
               << " " << parname << "_errsym=" << errsym 
               << " " << parname << "_errm=" << errm 
               << " " << parname << "_errp=" << errp 
               << " " << parname << "_globcor=" << globcor
               << " " << parname << "_step=" << fitstep
               << " " << parname << "_chi2=" << pchi2 ;
    }
    // fit theories
    for (par=gStore()->GetActiveParameters().begin(); par!=gStore()->GetActiveParameters().end(); ++par) {
      if ( !(*par)->HasTheory() ) continue; // parameters
      parname = (*par)->GetFullName(); //GetParName() ;
      parname.ReplaceAll(":","_");
      parname.ReplaceAll(".","_");
      val = (*par)->GetTheoryPrediction() ;
      init = (*par)->GetValue() ;
      fitstep = (*par)->GetFitStep() ;
      pchi2 = (*par)->GetChiSquared() ;

      m_logger << " " << parname << "=" << val
               << " " << parname << "_init=" << init
               << " " << parname << "_step=" << fitstep
               << " " << parname << "_chi2=" << pchi2 ;
    }

    m_logger << " " << GEndl;
    m_logger << kINFO << "End of " << summaryprefix << GEndl; // 'empty' line in summary
  }

  // Parameters to reactivate after this fit
  if (m_deactivated) reactivate();

  return m_chi2minLoop;
}

void Gfitter::GGSLMinimizer::DrawContour( GParameter* gpar1, GParameter* gpar2, 
                                          Int_t npoints, Double_t dchi2, Double_t* npx, Double_t* npy )
{
   // Parameters to deactivate for this fit?
   if (!m_deactivated) deactivate();

   // this is not very elegant... and even worse wrong if the active pars dont start at index 0
   int par1=-1, par2=-1;
   for (int ipar=0; ipar < GetNumOfFitPars(); ipar++) {
      if (GetFitPars()[ipar] == gpar1) par1 = ipar;
      if (GetFitPars()[ipar] == gpar2) par2 = ipar;
   }      
   if (par1 < 0 || par2 < 0) {
      m_logger << kFATAL << "Could not attribute contour parameters to fit parameters: "
               << gpar1->GetParName() << " " << gpar2->GetParName() << GEndl;
   }

   _theFitter->Config().MinimizerOptions().SetErrorDef(dchi2);

   //ROOT::Minimizer
   unsigned int upar1 = static_cast<unsigned int>(par1);
   unsigned int upar2 = static_cast<unsigned int>(par2);
   unsigned int unpoints = static_cast<unsigned int>(npoints);
   (void) _theFitter->GetMinimizer()->Contour(upar1,
                                              upar2,
                                              unpoints,
                                              npx,npy);

  // Parameters to reactivate after this fit
  if (m_deactivated) reactivate();
}

void Gfitter::GGSLMinimizer::UpdateResults()
{
   // the parameter vector
   const GParPtrVec_t& fp = GetFitPars();
   const UInt_t npar = fp.size();

   // retrieve covariance matrix first
   TMatrixD* covMat = new TMatrixD( npar, npar );
   for (UInt_t i=0; i < npar; i++) {
      for (UInt_t j=0; j < npar; j++) (*covMat)(i,j) = _theFitter->Result().CovMatrix(i,j);
   }
   
   for (UInt_t ipar=0; ipar < npar; ipar++) {

      Double_t val, errp, errm, errsym, globcor, chi2;
      val     = _theFitter->Result().Parameters()[ipar] ;
      errp    = _theFitter->Result().UpperError(ipar) ;
      errm    = _theFitter->Result().LowerError(ipar) ;
      errsym  = _theFitter->Result().Error(ipar) ;
      globcor = _theFitter->Result().GlobalCC(ipar) ;
      chi2    = _theFitter->Result().MinFcnValue() ;

      // NOTE: covariance matrix is now owned by GResultEval: all objects have the same pointer reference
      fp[ipar]->SetResult( new GResultEval( val, chi2, errp, errm, errsym, globcor, covMat, GetFitDuration() ) );
   }
}

void Gfitter::GGSLMinimizer::IFCN( Int_t& /* npars */, Double_t* /* grad */, Double_t &f, Double_t* fitPars, Int_t iflag )
{
   // Evaluate the minimisation function ----------------------------------------------------
   //
   //  Input parameters:
   //    npars:   number of currently variable parameters
   //             CAUTION: this is not (necessarily) the dimension of the fitPars vector !
   //    fitPars: array of (constant and variable) parameters
   //    iflag:   indicates what is to be calculated (see example below)
   //    grad:    array of gradients
   //
   //  Output parameters:
   //    f:       the calculated function value.
   //    grad:    the (optional) vector of first derivatives).
   // ---------------------------------------------------------------------------------------
   (GetThisPtr())->FCN( f, fitPars, iflag );
}

Gfitter::GGSLMinimizer* Gfitter::GGSLMinimizer::m_This = 0;

Gfitter::GGSLMinimizer* Gfitter::GGSLMinimizer::GetThisPtr()
{
   return m_This;
}

//_____________________________________________________________________________
void 
Gfitter::GGSLMinimizer::setMinimizerType(const char* type)
{
  // Choose the minimizer algorithm. 
  m_minimizerType = type; 
}

//_____________________________________________________________________________
void
Gfitter::GGSLMinimizer::setMinimizerAlg(const char* type)
{
  // Choose the minimizer algorithm. 
  m_minimizerAlg = type;
}

//_____________________________________________________________________________
void 
Gfitter::GGSLMinimizer::setEps(Double_t eps)
{
  // Change MINUIT tolerance
  m_eps = eps;
  if (_theFitter) { _theFitter->Config().MinimizerOptions().SetTolerance(m_eps); }
}

//_____________________________________________________________________________
void
Gfitter::GGSLMinimizer::runPreFitter()
{
  if (gStore()->GetPreFitter()) {
     gStore()->GetPreFitter()->Initialise();
     gStore()->GetPreFitter()->ExecuteFit();
  }
}

//_____________________________________________________________________________
double
Gfitter::GGSLMinimizer::runPreFitScan()
{
  std::vector<ROOT::Fit::ParameterSettings>& parameters = _theFitter->Config().ParamsSettings();
  Double_t chi2min = GetCurrentChi2();

  // apply pre-fit scan ?
  for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) {
    const GInterval* pfs = GetFitPars()[ipar]->GetPreFitScanRange();
    if (pfs) {
      Double_t bestc2 = GetCurrentChi2();
      Double_t bestv  = GetFitPars()[ipar]->GetFitValue();
      for (Int_t i=0; i<pfs->GetNsteps(); i++) {
        GetFitPars()[ipar]->SetFitValue( pfs->GetScanValue(i) );
        Double_t c2 = GetCurrentChi2();
        if (c2 < bestc2) { chi2min = bestc2 = c2; bestv = GetFitPars()[ipar]->GetFitValue(); }
      }
      GetFitPars()[ipar]->SetFitValue( bestv );
      parameters[ipar].SetValue( bestv ); // initialize the fit here
    }
  }

  // print summary
  m_logger << kINFO << "Best scan chi2 value : " << chi2min << GEndl;
  for (UInt_t ipar=0; ipar<GetFitPars().size(); ++ipar) {
    m_logger << kINFO << ">> Best scan parameter : " << GetFitPars()[ipar]->GetParName()
                      << " = " << GetFitPars()[ipar]->GetFitValue() << GEndl;
  }

  return chi2min;
}

//_____________________________________________________________________________
double
Gfitter::GGSLMinimizer::runPostFitScan()
{
  Double_t chi2min = GetCurrentChi2();

  // apply pre-fit scan ?
  for (Int_t ipar=0; ipar < GetNumOfFitPars(); ++ipar) {
    const GInterval* pfs = GetFitPars()[ipar]->GetPreFitScanRange();
    if (pfs) {
      Double_t bestc2 = GetCurrentChi2();
      Double_t bestv  = GetFitPars()[ipar]->GetFitValue();
      for (Int_t i=0; i<pfs->GetNsteps(); i++) {
        GetFitPars()[ipar]->SetFitValue( pfs->GetScanValue(i) );
        Double_t c2 = GetCurrentChi2();
        if (c2 < bestc2) { chi2min = bestc2 = c2; bestv = GetFitPars()[ipar]->GetFitValue(); }
      }
      m_bestvalsLoop[ipar] = bestv ;
    }
  }

  return chi2min;
}

//_____________________________________________________________________________
void
Gfitter::GGSLMinimizer::followUpFits()
{
   // improve minimum (returns 4 if no new minimum found -> this is NOT a problem !)
   if (m_useImprove) {
      _theFitter->Config().SetMinimizer(m_minimizerType.c_str(),"migradimproved");
      bool ret = _theFitter->FitFCN( *m_fcn );
      Int_t _status = ((ret) ? _theFitter->Result().Status() : -1);
      this->checkErr( "IMPROVE", _status, 0, -123, kTRUE );
   }

   // expensive MINOS analysis
   if (m_useMinos ) {
      Int_t _status = -1;
      if (_theFitter->GetMinimizer()==0) {
        m_logger << kWARNING << "Cannot run MINOS. Need to run MIGRAD first!" << GEndl ;
      } else {
        m_logger << kINFO << "Now running MINOS ..." << GEndl ;
        UpdateResults();
        bool ret = _theFitter->CalculateMinosErrors();
        _status = ((ret) ? _theFitter->Result().Status() : -1);
        m_logger << kINFO << "Finished running MINOS ..." << GEndl ;
      }
      this->checkErr( "MINOs", _status, 0, -123, kTRUE );
   }
}

//_____________________________________________________________________________
void
Gfitter::GGSLMinimizer::deactivate()
{
   if (m_deactivated) return; 

   GParPtrVec_t::iterator parIt = m_deactive.begin();
   for (; parIt!=m_deactive.end(); ++parIt)
     (*parIt)->SetActive(kFALSE);

   m_deactivated=true;
}

//_____________________________________________________________________________
void
Gfitter::GGSLMinimizer::reactivate()
{
  if (!m_deactivated) return; 

  GParPtrVec_t::iterator parIt = m_deactive.begin();
  for (; parIt!=m_deactive.end(); ++parIt)
    (*parIt)->SetActive(kTRUE);

  m_deactivated=false;
}

//_____________________________________________________________________________
void
Gfitter::GGSLMinimizer::doFit()
{
    // need to encapsulate the fit inside dedicated function b/c it applies a continue;
    // statement when done!

    // MB: hack, fit can be set unstable during fit to force exit of fit.
    this->SetFitUnstable(false);

    // minimize with desired algorithm, eg. MIGRAD
    _theFitter->Config().SetMinimizer(m_minimizerType.c_str(),m_minimizerAlg.c_str());
    bool ret = _theFitter->FitFCN( *m_fcn );
    Int_t _status = ((ret) ? _theFitter->Result().Status() : -1);
    this->checkErr( "MIGrad", _status, 0, -123, kTRUE );

    // MB: hack, reset unstable flag here, for next fit.
    this->SetFitUnstable(false);    
}
