/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GGeneticFitter                                                        *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 *      adapted from TMVA                                                         *
 *        [PoS ACAT {\bf }, 040 (2007), physics/0703039                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "Gfitter/GGeneticFitter.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTimer.h"

using namespace std;

Gfitter::GGeneticFitter::GGeneticFitter() 
   : GFitterBase(),
     m_ga(0)
{
   InitClassName( "GGeneticFitter" );
}

Gfitter::GGeneticFitter::~GGeneticFitter()
{
   if (m_ga) delete m_ga;
}

void Gfitter::GGeneticFitter::Initialise()
{
   m_logger << kINFO << "Initialise" << GEndl;
   
   m_This = this;      
   
   // base class call
   GFitterBase::Initialise();  

   this->ReadGAConfiguration();
   this->PrintGAConfiguration();

   // delete TFitter instance if existing
   delete m_ga; m_ga = 0; 

   // reset vectors              
   GetFitPars().clear();
   GetPenalisedPars().clear();

   // define fit parameters      
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
   vector<TMVA::Interval*> ranges;  
   for (; par != gStore()->GetActiveParameters().end(); par++) {
      if ((*par)->VariesInFit()) {
         
         // save name and pointer to parameter that is free in the fit
         GetFitPars().push_back( *par );
         
         // sanity check
         if ((*par)->GetFitStep() <= 0) m_logger << "Huge troubles with fit step of par: " 
                                                 << (*par)->GetParName() << GEndl;
         
         // book the fit parameter
         if ( (*par)->GetFitRange().GetMin() ) {
            ranges.push_back( new TMVA::Interval( (*par)->GetFitRange().GetMin(),
                                                    (*par)->GetFitRange().GetMax() ) );
         }
         else {
            ranges.push_back( new TMVA::Interval( (*par)->GetValue() - 10 * (*par)->GetData().GetErrTotm(),
                                                    (*par)->GetValue() + 10 * (*par)->GetData().GetErrTotm() ) ); 
         }
      }      
   }
   
   for (par = gStore()->GetParameters().begin(); par != gStore()->GetParameters().end(); par++) {
      if ((*par)->HasTheory() && (*par)->IsScanned() && (*par)->IsFirst()) {
         m_logger << kINFO << "Setting penalised Parameter: " << (*par)->GetParName() << GEndl;
         GetPenalisedPars().push_back( (*par) );
      }
   }        
   
   this->SetNumOfFitPars( ranges.size());

   // init covariance matrices and/or diagonal errors
   if (gStore()->IsCorrelationActive()) this->InitErrorMatrix();

   this->PrintConfiguration();

   m_ga = new TMVA::GeneticAlgorithm( (*(IFitterTarget*)GetThisPtr()) , m_populationSize, ranges );
}

Double_t Gfitter::GGeneticFitter::GetCurrentChi2()
{
   m_logger << kFATAL << "<GetCurrentChi2> Function not implemented for GeneticFitter" << GEndl;
   return 0;
}

Double_t Gfitter::GGeneticFitter::ExecuteFit()
{   
   m_logger << kINFO << "ExecuteFit" << GEndl;

   // init timer
   GTimer timer;

   do {
      m_ga->Init();
      m_ga->CalculateFitness();
      m_ga->GetGeneticPopulation().TrimPopulation();
      m_ga->SpreadControl( m_scsteps, m_scrate, m_scfactor );

   } while (!m_ga->HasConverged( m_convsteps, m_convcrit ));

   m_fitDuration = timer.GetElapsedTime();

   TMVA::GeneticGenes*   genes = m_ga->GetGeneticPopulation().GetGenes( 0 );
   std::vector<Double_t> gvec  = genes->GetFactors();
   std::vector<TString>  svec;

   m_logger << kINFO << "Elapsed time: " << GetFitDuration() << GEndl;
   // change for ROOT v5.24
   //m_logger << kINFO << "GA Results (Fitness = " << m_ga->GetGeneticPopulation().GetFitness() << "): " << GEndl;
   m_logger << kINFO << "GA Results (Fitness = " 
            << m_ga->GetGeneticPopulation().GetGenes(0)->GetFitness() << "): " << GEndl;

   // update parameters with these results
   for (UInt_t ipar=0; ipar<GetFitPars().size(); ipar++) {
      GetFitPars()[ipar]->SetFitValue( gvec[ipar] );
      svec.push_back( GetFitPars()[ipar]->GetParName() );
   }
   GUtils::FormattedOutput( gvec, svec, "Parameters", "Best values", m_logger, "%+1.4g" );
   m_logger << GEndl;

   return 0.;
}

void Gfitter::GGeneticFitter::DrawContour( GParameter*, GParameter*, 
                                           Int_t, Double_t, Double_t*, Double_t* )
{
   m_logger << kFATAL << "DrawContour not implemented for GGeneticFitter" << GEndl;
}

void Gfitter::GGeneticFitter::UpdateResults()
{
   m_logger << kFATAL << "UpdateResults not implemented for GGeneticFitter" << GEndl;
}

Gfitter::GGeneticFitter* Gfitter::GGeneticFitter::m_This = 0;

Gfitter::GGeneticFitter* Gfitter::GGeneticFitter::GetThisPtr()
{
   return m_This;
}

Double_t Gfitter::GGeneticFitter::EstimatorFunction( std::vector<Double_t>& parameters )
{ 
   const Int_t npars = parameters.size();
   Double_t    fitPars[npars];
   for (UInt_t i=0; i<parameters.size(); i++) fitPars[i] = parameters[i];
   
   Double_t f = 0.;
   Int_t iflag = 0;
   
   ((GGeneticFitter*)GetThisPtr())->FCN( f, fitPars, iflag );
   
   return f; 
} 

void Gfitter::GGeneticFitter::ReadGAConfiguration()
{ 
   // read and set GA parameters
   m_populationSize = (gStore()->ExistVariable( "GeneticFitter::PopulationSize" ) ? 
                       gStore()->GetVariable( "GeneticFitter::PopulationSize" )->GetIntValue() : 100);
   
   m_convsteps      = (gStore()->ExistVariable( "GeneticFitter::ConvSteps" ) ? 
                       gStore()->GetVariable( "GeneticFitter::ConvSteps" )->GetIntValue() : 20);
   
   m_convcrit       = (gStore()->ExistVariable( "GeneticFitter::ConvCrit" ) ? 
                       gStore()->GetVariable( "GeneticFitter::ConvCrit" )->GetFloatValue() : 0.0001);
   
   m_scsteps        = (gStore()->ExistVariable( "GeneticFitter::ScSteps" ) ? 
                       gStore()->GetVariable( "GeneticFitter::ScSteps" )->GetIntValue() : 10);
   
   m_scrate         = (gStore()->ExistVariable( "GeneticFitter::ScRate" ) ? 
                       gStore()->GetVariable( "GeneticFitter::ScRate" )->GetIntValue() : 5);
   
   m_scfactor       = (gStore()->ExistVariable( "GeneticFitter::ScFactor" ) ? 
                       gStore()->GetVariable( "GeneticFitter::ScFactor" )->GetFloatValue() : 0.95);
}

void Gfitter::GGeneticFitter::PrintGAConfiguration()
{ 
   m_logger << kINFO << "GA CONFIGURATION : " << GEndl;
   m_logger << kINFO << "  PopulationSize : " << m_populationSize << GEndl;
   m_logger << kINFO << "  ConvSteps      : " << m_convsteps << GEndl;
   m_logger << kINFO << "  ConvCrit       : " << m_convcrit  << GEndl;
   m_logger << kINFO << "  ScSteps        : " << m_scsteps   << GEndl;
   m_logger << kINFO << "  ScRate         : " << m_scrate    << GEndl;
   m_logger << kINFO << "  ScFactor       : " << m_scfactor  << GEndl;
}
