/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GMath                                                                 *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>

#include "TMath.h"

#include "Gfitter/GMath.h"

// matrix owned by caller
TMatrixD* Gfitter::GMath::GetCorrelationMatrix( const TMatrixD& covMat )
{
   // sanity check
   Int_t nvar = covMat.GetNrows();
   if (nvar != covMat.GetNcols()) {
      std::cout << "<GMath::GetCorrelationMatrix> Fatal error: input matrix not quadratic" << std::endl;
      exit(1);
   }
   
   TMatrixD* corrMat = new TMatrixD( nvar, nvar );

   for (Int_t ivar=0; ivar<nvar; ivar++) {
      for (Int_t jvar=0; jvar<nvar; jvar++) {
         if (ivar != jvar) {
            Double_t d = covMat(ivar, ivar)*covMat(jvar, jvar);
            if (d > 0) (*corrMat)(ivar, jvar) = covMat(ivar, jvar)/TMath::Sqrt(d);
            else {
               std::cout << "<GMath::GetCorrelationMatrix> Warning: zero variances for variables "
                         << "(" << ivar << ", " << jvar << ")" << std::endl;
               (*corrMat)(ivar, jvar) = 0;
            }
         }
         else (*corrMat)(ivar, ivar) = 1.0;
      }
   }

   return corrMat;
}

//
Double_t Gfitter::GMath::AsymGauss(Double_t *x, Double_t *params)
{
   Double_t mean = params[0];
   Double_t errp = params[1];
   Double_t errm = params[2];

   Double_t sigma=0.;

   if ( x[0] < mean)
      sigma = errm;
   else 
      sigma = errp;

   return TMath::Gaus(x[0], mean, sigma);
}
