/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GMsgLogger                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      output logger class producing nicely formatted log messages               *
 *                                                                                *
 *      Adapted from TMVA:MsgLogger. Original authors:                            *
 *                                                                                *
 *      Attila Krasznahorkay <Attila.Krasznahorkay@cern.ch> - CERN, Switzerland   *
 *      Andreas Hoecker       <Andreas.Hocker@cern.ch> - CERN, Switzerland        *
 *      Joerg Stelzer         <stelzer@cern.ch>        - DESY, Germany            *
 *      Eckhard v. Toerne     <evt@uni-bonn.de>        - U of Bonn, Germany       *
 *                                                                                *
 * Copyright (c) 2005-2011:                                                       *
 *      CERN, Switzerland                                                         *
 *      U. of Victoria, Canada                                                    *
 *      MPI-K Heidelberg, Germany                                                 *
 *      U. of Bonn, Germany                                                       *
 *                                                                                *
 *   http://root.cern.ch/root/htmldoc/src/TMVA__MsgLogger.h.html                  *
 *                                                                                *
 * (http://tmva.sourceforge.net/LICENSE)                                          *
 **********************************************************************************/


#ifndef ROOT_Gfitter_GMsgLogger
#define ROOT_Gfitter_GMsgLogger

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// GMsgLogger                                                           //
//                                                                      //
// ostreamstream derivative to redirect and format output               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

// STL include(s):
#include <string>
#include <sstream>
#include <map>

// ROOT include(s)
#include "TObject.h"
#include "TString.h"

// Local include(s):

namespace Gfitter {

   // forward declarations
   class GStore;

   // define outside of class to facilite access
   enum GMsgLevel { 
      kVERBOSE = 1, 
      kDEBUG   = 2,
      kINFO    = 3,
      kWARNING = 4,
      kERROR   = 5,
      kFATAL   = 6,
      kALWAYS  = 7
   };

   class GMsgLogger : public std::ostringstream, public TObject {

   public:

      GMsgLogger( const TObject* source, GMsgLevel minLevel = kINFO );
      GMsgLogger( const std::string& source, GMsgLevel minLevel = kINFO );
      GMsgLogger( GMsgLevel minLevel = kINFO );
      GMsgLogger( const GMsgLogger& parent );
      ~GMsgLogger();

      // Accessors
      void               SetSource( const char*        source ) { m_strSource = source; }
      void               SetSource( const std::string& source ) { m_strSource = source; }
      void               SetSource( const TString&     source ) { m_strSource = source.Data(); }
      const std::string& GetSource() const { return m_strSource; }

      UInt_t      GetMaxSourceSize() const       { return (UInt_t)m_maxSourceSize; }
      std::string GetPrintedSource() const;
      std::string GetFormattedSource() const;

      GMsgLevel          GetMinLevel() const     { return m_minLevel; }
      const std::string& GetMinLevelStr() const  { return m_levelMap.find( m_minLevel )->second; }

      GMsgLevel   MapLevel( const TString& instr ) const;
      
      // Needed for copying
      GMsgLogger& operator= ( const GMsgLogger& parent );

      // Stream modifier(s)
      static GMsgLogger& endmsg( GMsgLogger& logger );
      
      // Accept stream modifiers
      GMsgLogger& operator<< ( GMsgLogger& ( *_f )( GMsgLogger& ) );
      GMsgLogger& operator<< ( std::ostream& ( *_f )( std::ostream& ) );
      GMsgLogger& operator<< ( std::ios& ( *_f )( std::ios& ) );
      
      // Accept message level specification
      GMsgLogger& operator<< ( GMsgLevel level );
      
      // For all the "conventional" inputs
      template <class T> GMsgLogger& operator<< ( T arg ) {
         *(std::ostringstream*)this << arg; return *this;
      }

   private:

      // the current minimum level is global for the whole Gfitter
      // it can only be changed by the central storage singleton object
      friend class GStore;
      static void SetMinLevel( GMsgLevel minLevel ) { m_minLevel = minLevel; }
      static GMsgLevel                 m_minLevel;       // minimum level for logging output

      // private utility routines
      void Send();
      void InitMaps();
      void WriteMsg( GMsgLevel level, const std::string& line ) const;

      const TObject*                   m_objSource;      // the source TObject (used for name)
      std::string                      m_strSource;      // alternative string source
      const std::string                m_prefix;         // the prefix of the source name
      const std::string                m_suffix;         // suffix following source name
      GMsgLevel                        m_activeLevel;    // active level
      const std::string::size_type     m_maxSourceSize;  // maximum length of source name

      std::map<GMsgLevel, std::string> m_levelMap;       // matches output levels with strings
      std::map<GMsgLevel, std::string> m_colorMap;       // matches output levels with terminal colors

   }; // class GMsgLogger

   inline GMsgLogger& GMsgLogger::operator<< ( GMsgLogger& (*_f)( GMsgLogger& ) ) 
   {
      return (_f)(*this);
   }

   inline GMsgLogger& GMsgLogger::operator<< ( std::ostream& (*_f)( std::ostream& ) ) 
   {
      (_f)(*this);
      return *this;
   }

   inline GMsgLogger& GMsgLogger::operator<< ( std::ios& ( *_f )( std::ios& ) ) 
   {
      (_f)(*this);
      return *this;
   }

   inline GMsgLogger& GMsgLogger::operator<< ( GMsgLevel level ) 
   {
      m_activeLevel = level;
      return *this;
   }

   // Although the proper definition of "Endl" as a function pointer
   // would be nicer C++-wise, it introduces some "unused variable"
   // warnings so let's use the #define definition after all...
   // [ static GMsgLogger& ( *Endl )( GMsgLogger& ) = &GMsgLogger::endmsg; ]
#define GEndl GMsgLogger::endmsg

}

#endif // Gfitter_GMsgLogger
