/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GParameter                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <iostream>

#include "TString.h"
#include "TFormula.h"
#include "TMath.h"

#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GTheoryFactory.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GDataBase.h"
#include "Gfitter/GData.h"
#include "Gfitter/GDataCurve.h"
#include "Gfitter/GWorkspace.h"
#include "Gfitter/GResultBase.h"

ClassImp(Gfitter::GParameter)

using namespace std;

const Double_t Gfitter::GParameter::m_nsig = 10.0;

Gfitter::GParameter::GParameter()
   : GObject(),
     m_parName ( "" ),
     m_hasAlias( kFALSE ),
     m_parAlias( "" ),
     m_data    ( 0 ),
     m_refdata ( 0 ),
     m_active  ( kFALSE ),
     m_fitStep ( 0 ),
     m_fitRange( 0 ),
     m_scanRange( 0 ),
     m_nbins( 100 ),
     m_preFitScanRange( 0 ),
     m_ModifyInToy( 0 ),
     m_theory  ( 0 ),
     m_parBackup( 0 ),
     m_scanned ( kFALSE ),
     m_clientReferences( new TList() ),
     m_result ( 0 ),
     m_isFirst( kTRUE ),
     m_compValCache  ( 0 ),
     m_hasTheoryCache( kFALSE )
{
   InitClassName( "GParameter" );
}

Gfitter::GParameter::GParameter( TString parName, TString parAlias, Bool_t active, 
                                 GDataBase* data, GInterval* scanRange, TString theoryName )
   : GObject(),
     m_parName ( parName ),
     m_hasAlias( parName != parAlias ),
     m_parAlias( GUtils::RemoveNamespace( parAlias ) ),
     m_data    ( data ),
     m_refdata ( 0 ),
     m_active  ( active ),
     m_fitStep ( 0 ),
     m_fitRange( 0 ),
     m_scanRange( scanRange ),
     m_preFitScanRange( 0 ),
     m_ModifyInToy( 0 ),
     m_theory  ( 0 ),
     m_parBackup( 0 ),
     m_scanned ( kFALSE ),
     m_clientReferences( new TList() ),
     m_result ( 0 ),
     m_isFirst( kTRUE ),
     m_compValCache  ( 0 ),
     m_hasTheoryCache( kFALSE )
{
   InitClassName( "GParameter" );
   
   // sanity checks
   if (m_data == 0) m_logger << kFATAL << "Zero \"data\" pointer" << GEndl;
   
   // different initialisation for GData and GDataCurve
   GData* tmpData = dynamic_cast<GData*>(m_data);
   GDataCurve* tmpDataCurve = dynamic_cast<GDataCurve*>(m_data);
   GWorkspace* tmpWorkspace = dynamic_cast<GWorkspace*>(m_data);

  if (tmpData != 0) { //it is a GData

      // set reasonable scan range if not defined
      if (m_scanRange == 0) {
         m_scanRange = new GInterval( tmpData->GetValue() - 4*tmpData->GetErrTotm(),
                                      tmpData->GetValue() + 4*tmpData->GetErrTotp() );
      }

      // init the fit limits
      InitFitRanges();
      
      m_curve = kFALSE;
  } 
  else if (tmpDataCurve!=0) { // it is a GDataCurve
    // set reasonable scan range if not defined    
      if (m_scanRange == 0) {
         m_scanRange = new GInterval( tmpDataCurve->GetMin(), tmpDataCurve->GetMax() );
      }

      // set reasonable fit range
      m_fitRange = new GInterval( tmpDataCurve->GetMin(), tmpDataCurve->GetMax() );
      m_fitStep  = m_fitRange->GetWidth()/10.0;

      m_curve = kTRUE;
  }
  else if (tmpWorkspace!=0) {
    // MB: Needed here?
    // set reasonable scan range if not defined    
    if (m_scanRange == 0) {
      m_scanRange = new GInterval( );
    }
    // set reasonable fit range
    m_fitRange = new GInterval( );
    m_fitStep  = 0.5;
    m_curve = kTRUE;    
  }
   
   // common default initialisation
   this->InitTheory( theoryName );

}
Gfitter::GParameter::GParameter( TString parName, TString parAlias, Bool_t active, 
                                 GInterval* validityRange, GInterval* scanRange, TString theoryName )
   : GObject(),
     m_parName ( parName ),
     m_hasAlias( parName != parAlias ),
     m_parAlias( GUtils::RemoveNamespace( parAlias ) ),
     m_active  ( active ),
     m_fitStep ( 0 ),
     m_fitRange( 0 ),
     m_scanRange( scanRange ),
     m_preFitScanRange( 0 ),
     m_ModifyInToy( 0 ),
     m_theory  ( 0 ),
     m_parBackup( 0 ),
     m_scanned ( kFALSE ),
     m_curve ( kFALSE ),
     m_clientReferences( new TList() ),
     m_isFirst( kTRUE ),
     m_compValCache  ( 0 ),
     m_hasTheoryCache( kFALSE )
{
   InitClassName( "GParameter" );

   // sanity checks
   if (validityRange == 0) m_logger << kFATAL << "Zero \"validityRange\" pointer" << GEndl;

   // set reasonable scan range if not defined (equal to validity range)
   if (m_scanRange == 0) m_scanRange = new GInterval( *validityRange );

   // set the fit range
   m_fitStep  = validityRange->GetWidth()/100.0;
   m_fitRange = new GInterval( *validityRange );
  
   // create value and errors
   m_data = new GData( *validityRange );

   // common default initialisation
   InitTheory( theoryName );
}
  
Gfitter::GParameter::GParameter( const GParameter& par )
   : GObject( par ),
     m_data     ( 0 ),
     m_refdata  ( 0 ),
     m_fitStep  ( 0 ),
     m_fitRange ( 0 ),
     m_scanRange( 0 ),
     m_preFitScanRange( 0 ),
     m_theory   ( 0 ),
     m_parBackup( 0 ),
     m_clientReferences( new TList() ),
     m_result   ( 0 ),
     m_compValCache  ( 0 ),
     m_hasTheoryCache( 0 )
{
   InitClassName( "GParameter" );

   CopyPar( par );
}

Gfitter::GParameter::~GParameter()
{
   if (m_data             != 0) { delete m_data;            m_data = 0; }
   if (m_refdata          != 0) { delete m_refdata;         m_refdata = 0; }
   if (m_result           != 0) { delete m_result;          m_result = 0; }
   if (m_fitRange         != 0) { delete m_fitRange;        m_fitRange = 0; }
   if (m_scanRange        != 0) { delete m_scanRange;       m_scanRange = 0; }
   if (m_preFitScanRange  != 0) { delete m_preFitScanRange; m_preFitScanRange = 0; }
   /// MB: All theories are deleted in GTheoryFactory::Reset(), where they are owned.
   ///     This function is called in the destructor of GStore.
   //if (m_theory           != 0) { delete m_theory;          m_theory = 0; }
   m_theory = 0;
   if (m_parBackup        != 0) { delete m_parBackup;       m_parBackup = 0; }
   if (m_clientReferences != 0) { 
      m_clientReferences->Clear(); 
      delete m_clientReferences; m_clientReferences = 0; 
   }
}

void Gfitter::GParameter::CopyPar( const GParameter& other )
{
   m_parName    = other.GetParName();
   m_hasAlias   = other.HasAlias();
   m_parAlias   = other.GetParAlias();
   if (m_data    != 0) { delete m_data;    m_data = 0; }
   if (m_refdata != 0) { delete m_refdata; m_refdata = 0; }
 
   // copy data
   if (other.GetDataPtr()) {
      GData* tmpData = dynamic_cast<GData*>(other.GetDataPtr());
      if (!other.IsCurve()) { //it's a GData!
         m_data = new GData( *tmpData  );
      } 
      else {
	GDataCurve* tmpDataCurve = dynamic_cast<GDataCurve*>(other.GetDataPtr());
	if (tmpDataCurve!=0) m_data = new GDataCurve( *tmpDataCurve  );
	else {
	  GWorkspace* tmpWorkspace = dynamic_cast<GWorkspace*>(other.GetDataPtr());
	  if (tmpWorkspace!=0) m_data = new GWorkspace( *tmpWorkspace  );
	}
      }
   }
   
   // copy reference data (if exists)
   if (other.GetRefDataPtr()) {
      GData* tmpData = dynamic_cast<GData*>(other.GetRefDataPtr());
      if (!other.IsCurve()) { //it's a GData!
         m_refdata = new GData( *tmpData  );
      } 
      else {
         GDataCurve* tmpDataCurve = dynamic_cast<GDataCurve*>(other.GetRefDataPtr());
         if (tmpDataCurve!=0) m_refdata = new GDataCurve( *tmpDataCurve  );
	 else {
	   GWorkspace* tmpWorkspace = dynamic_cast<GWorkspace*>(other.GetRefDataPtr());
	   if (tmpWorkspace!=0) m_data = new GWorkspace( *tmpWorkspace  );
	 }
      }
   }
 
   m_active      = other.IsActive();
   m_ModifyInToy = other.GetModifyInToy();
   m_fitStep     = other.GetFitStep();
   if (m_fitRange != 0) delete m_fitRange;
   m_fitRange    = new GInterval( other.GetFitRange() );
   if (m_scanRange != 0) delete m_scanRange;
   m_scanRange   = new GInterval( other.GetScanRange() );
   if (m_preFitScanRange != 0) delete m_preFitScanRange;
   if (other.GetPreFitScanRange()) m_preFitScanRange = new GInterval( *other.GetPreFitScanRange() );
   else m_preFitScanRange = 0;

   m_theoryName  = other.GetTheoryName();
   m_theory      = other.GetTheory();	

   m_parBackup   = other.GetBackup();
   m_scanned     = other.IsScanned();
   m_curve       = other.IsCurve();
   m_isFirst     = other.IsFirst();

   m_compValCache   = other.GetCachedCompValue();
   m_hasTheoryCache = other.GetCachedHasTheory();

   m_samePars.resize( other.GetSameParameters().size() );
   std::copy( other.GetSameParameters().begin(), other.GetSameParameters().end(), m_samePars.begin() );
 
   m_clientReferences->Clear(); 
   m_clientReferences->AddAll( &other.GetClientReferenceList() );
}

void Gfitter::GParameter::SetActive( Bool_t active ) 
{ 
   m_active = active;
  
   if (m_theory == 0) {
      m_logger << kDEBUG 
               << "<SetActive> Need a new initialisation of the fitter, when this function is" 
               << " called for a parameter without respective theory ... please check!" << GEndl;
   }
}

void Gfitter::GParameter::SetFitRange( GInterval* f ) 
{ 
   if (m_fitRange) delete m_fitRange;
   m_fitRange = f; 
}

void Gfitter::GParameter::SetPreFitScanRange( GInterval* f ) 
{ 
   if (m_preFitScanRange) delete m_preFitScanRange;
   m_preFitScanRange = f; 
   if (!m_preFitScanRange) m_logger << kFATAL << "<SetPreFitScanRange> Zero pointer given" << GEndl;

   // sanity check
   if (GetFitRange().IsActive() && !GetFitRange().InInterval( *m_preFitScanRange )) {
      m_logger << kWARNING << "<SetPreFitScanRange> prefit interval exceeds fit ranges: " 
               << *m_preFitScanRange << " > " << GetFitRange() << " reset limits" << GEndl;
      Double_t min    = TMath::Max( m_preFitScanRange->GetMin(), GetFitRange().GetMin() );
      Double_t max    = TMath::Min( m_preFitScanRange->GetMax(), GetFitRange().GetMax() );
      Int_t    nsteps = m_preFitScanRange->GetNsteps();
      delete m_preFitScanRange;
      m_preFitScanRange = new GInterval( min, max, nsteps );
   }
}

void Gfitter::GParameter::SetRefData( GDataBase* d ) 
{ 
   m_refdata = d; 
}

void Gfitter::GParameter::InitTheory( TString theoryName )
{
   // default initialisation

   // only initialise theory if parameter active

   // JH: initiliase all: needed for scan!
   //if (!IsActive()) return;
   
   // set theory name
   if (theoryName == "same") m_theoryName = m_parName;
   else                      m_theoryName = theoryName;

   // retrieve corresponding theory
   GTheoryBase* tmpTh = GTheoryFactory::GetTheory( m_theoryName );
   if (tmpTh != 0) {
      // the theory must be of type GTheory (one that is defined in the DataCard)
      // An auxiliary theory can not be used
      SetTheory( dynamic_cast<GTheory*>(tmpTh) );
      if (GetTheory() == 0) {
         m_logger << kFATAL << "<InitTheory> The requested theory \"" << m_theoryName 
                  << "\" is not defined in the datacard" << GEndl;
      }
   }
}

void Gfitter::GParameter::UpdatePersistencyCache() 
{ 
   // update persistent values
   m_compValCache   = (HasTheory() ? GetTheory()->GetPrediction() : GetFitValue()); 
   m_hasTheoryCache = HasTheory();
}

void Gfitter::GParameter::InitFitRanges()
{
   // reset fit ranges only if data object is GData (as opposed to a data curve)
   GData* tmpData = dynamic_cast<GData*>(m_data);
   if (tmpData != 0) { //it is a GData

      if (m_fitRange != 0) delete m_fitRange;
      
      // set reasonable fit range
      if (tmpData->GetErrGaussm() != 0 || HasScalers()) {
         m_fitRange = new GInterval( 0, 0 ); // no limits (preferred by Minuit)
         m_fitStep  = 0.5*tmpData->GetErrTotSym();

         // print warning message: a parameter with solely a theoretical error cannot 
         // (usually) have scalers, since the chi2 function will have a discontinuity at 
         // the border of the theoretically limited region -> the Minuit fit will fail !
         if (tmpData->GetErrGaussSym() <= 0) {
            m_logger << kWARNING << "Parameter: \"" << GetFullName() << "\" contains scalers "
                     << "but has only theory errors: this creates a discontinuity in the chi-squared "
                     << "function, which will lead to a fit failure --> adding a small statistical "
                     << "error will remediate the problem."
                     << GEndl;
         }
      }
      else {                  
         m_fitRange = new GInterval(  tmpData->GetValue() - tmpData->GetErrRangem(), 
                                      tmpData->GetValue() + tmpData->GetErrRangep() );         
         m_fitStep  = m_fitRange->GetWidth()/10.0;
      }
   }
}

Bool_t Gfitter::GParameter::LinkIfSameParameter( GParameter* gpar )
{
   // sanity checks first
   // can only add parameter if this one IsFirst
   if (!this->IsFirst()) return kFALSE;

   // can only be added if exists and is different from this one
   if (gpar == 0 || gpar == this) return kFALSE;

   // both, this and the other parameter must be active
   if (!IsActive() || !gpar->IsActive()) return kFALSE;

   // can only be added if same name (NOTE: alias could be different !)
   if (*this != *gpar) return kFALSE;

   // can only be added if not yet in list
   GParPtrVec_t::iterator par = std::find(m_samePars.begin(), m_samePars.end(), gpar);
   if (par != m_samePars.end()) return kFALSE;

   // finally add to list
   m_samePars.push_back( gpar );
   
   return kTRUE;
}

Double_t Gfitter::GParameter::GetDeviation( Int_t& sign, TFormula* formula ) const
{
   const GData* tmpData = dynamic_cast<GData*>(m_data);
   
   if (tmpData == 0) { // it is GDataCurve 
      m_logger << kFATAL << "<GetDeviation> Parameter with dataCurve or workspace has no deviation!" 
               << GetFullName() << GEndl;; 
   }
   
//    if (HasTheory() ) 
//       m_logger << "has theory --> GetPrediction" << GEndl;
//    else
//       m_logger << "has no theory --> GetFitValue" << GEndl;
   
   // default return value
   Double_t compval  = HasTheory() ? GetTheory()->GetPrediction() : GetFitValue();   

   // theory range
   Double_t v    = GetValue( *tmpData );
   Double_t rmin = v - tmpData->GetErrRangem();
   Double_t rmax = v + tmpData->GetErrRangep();

   // compute chi-squared a la Rfit, assuming for the time being 
   // that parameters are unbound and uncorrelated
   sign = -1; // used as integer flag to indicate non-zero deviation
   Double_t retval  = 0.0;
   if (compval < rmin) {
      sign   = -1;
      if (formula) retval = formula->Eval(compval) - formula->Eval(rmin); 
      else         retval = (compval - rmin);
   }         
   else if (compval > rmax) {
      sign   = +1;
      if (formula) retval = formula->Eval(compval) - formula->Eval(rmax); 
      else         retval = (compval - rmax);
   }
   // else: parameter within bounds of theory error --> no effective delta

   // m_logger << kINFO << GetParName() << " : " << compval << " : " 
   //          << v << " : " << rmin << " : " << rmax << " : " << sign << " : " << retval << GEndl;
   
   return retval;
}

Double_t Gfitter::GParameter::GetChiSquared() const
{
   GData* tmpData = dynamic_cast<GData*>(m_data);
   Double_t retval( 0 );
   if (tmpData != 0) { //it is a GData    
      Int_t    sign( 0 );
      Double_t delta = GetDeviation( sign );
      
      if (delta == 0) retval = 0; // within theory bounds
      
      // compute chi-squared a la Rfit, assuming for the time being 
      // that parameters are unbound and uncorrelated
      
      else if (sign < 0) retval = delta/GetErrGaussm()*delta/GetErrGaussm();
      else if (sign > 0) retval = delta/GetErrGaussp()*delta/GetErrGaussp();
   }
   else { // it is a GDataCurve or GWorkspace
      Double_t fitValue = HasTheory() ? GetTheory()->GetPrediction() : GetFitValue(); 

      GDataCurve* tmpDataCurve = dynamic_cast<GDataCurve*>(m_data);
      if (tmpDataCurve!=0) {
	retval = tmpDataCurve->GDataCurve::GetChiSquared(fitValue);
	return retval;
      }

      GWorkspace* tmpWorkspace = dynamic_cast<GWorkspace*>(m_data);
      if (tmpWorkspace!=0) {
	retval = tmpWorkspace->GWorkspace::GetChiSquared();
	return retval;
      }
   }

   return retval;
}

/*
Double_t Gfitter::GParameter::GetChiSquared() const
{
   GData* tmpData = dynamic_cast<GData*>(m_data);
   if (tmpData != 0) { //it is a GData    
      Int_t    sign( 0 );
      Double_t delta = GetDeviation( sign );
      if (delta == 0) return 0; // within theory bounds
      
      // compute chi-squared a la Rfit, assuming for the time being 
      // that parameters are unbound and uncorrelated
      Double_t retval( 0 );
      m_logger << kINFO  << " : " << tmpData->GetErrGaussm() << GEndl;
      if      (sign < 0) retval = delta/GetErrGaussm();
      else if (sign > 0) retval = delta/GetErrGaussp();

      return retval*retval;
   }
   else { // it is a GDataCurve
      GDataCurve* tmpDataCurve = dynamic_cast<GDataCurve*>(m_data);
      return tmpDataCurve->GDataCurve::GetChiSquared();
   }

   return 0.;
}
*/
Double_t Gfitter::GParameter::GetPenaltyChiSquared() const
{
   // this is only possible if parameter has theory
   if (!HasTheory()) {
      m_logger << kFATAL << "<GetPenaltyChiSquared> Parameter: \"" << GetFullName() 
               << "\" is required a penalty while it does not have an associated theory" << GEndl;
   }

   // the penalty should be large compared to the bin width
   Double_t penaltyFactor = gStore()->GetPenaltyFactor();
   if (m_nbins <= 0) {
      m_logger << kFATAL << "<GetPenaltyChiSquared> Parameter: \"" 
               << GetFullName() << "\" invalid number of bins: "
               << m_nbins << GEndl;
   }
   if (GetScanRange().GetWidth() <= 0) {
      m_logger << kFATAL << "<GetPenaltyChiSquared> Parameter: \"" 
               << GetFullName() << "\" invalid scan range: "
               << GetScanRange() << GEndl;
   }         
   
   m_logger << kDEBUG << "<GetPenaltyChiSquared> Parameter: \"" 
            << GetFullName() << "\" penaltyfactor: " << penaltyFactor 
            << "\" bins: " << m_nbins << GEndl;

   Double_t sigma = penaltyFactor*GetScanRange().GetWidth()/m_nbins;
   Double_t delta = (GetTheory()->GetPrediction() - GetFitValue())/sigma;

   return delta*delta;
}

void Gfitter::GParameter::Backup()
{
   // copy parameter into m_backup
   if (m_parBackup != 0) delete m_parBackup;
   m_parBackup = new GParameter( *this );
}

void Gfitter::GParameter::Recover()
{
   // recover parameter from m_backup
   if (!m_parBackup) m_logger << kFATAL << "<Recover> Backup does not exist!" << GEndl;
   this->CopyPar( *m_parBackup );
   if (m_parBackup != 0) { delete m_parBackup; m_parBackup = 0; }
}

void Gfitter::GParameter::RegisterClientReference( GParameterRef* parRef ) 
{
   if (m_clientReferences->FindObject(parRef) != 0) {
      m_logger << kWARNING << "The parameter \"" 
               << GetParName() << "\" already has the theory \""
               << parRef->GetTheoryName() << "\" as a client" << GEndl;
      return;
   }
   m_clientReferences->Add(parRef);
}
   
void Gfitter::GParameter::NotifyReferencesOfChange() 
{
   //   m_logger << kINFO << "inside NotifyReferencesOfChange" << GEndl; 
   TListIter clIt( m_clientReferences );
   while (GParameterRef* parRef = (GParameterRef*)clIt()) {
      //      m_logger << "SetChanged for: " << (parRef)->GetTheoryName() << GEndl;
      parRef->SetChanged();
   }
}

Double_t Gfitter::GParameter::GetTheoryPrediction() const 
{ 
   if (!HasTheory()) m_logger << kFATAL << "Parameter \"" << GetFullName() << "\" has no theory!" << GEndl;

   Double_t retval = GetTheory()->GetPrediction();
   if (TMath::IsNaN( retval )) {
      m_logger << kFATAL << "NAN value returned from theory: " << GetTheory()->GetTheoryName() << GEndl;    
   }

   return retval;
}

Double_t Gfitter::GParameter::GetValue( const GDataBase& data ) const 
{
   Double_t retval = data.GetValue();

   // rescale reference value in case of scalers
   if (HasScalers()) {
      for (GSclPtrVec_t::const_iterator it = m_scalers.begin(); it != m_scalers.end(); it++) {
         retval += (*it)->GetBias();
      }
   }

   if (!GetFitRange().InInterval( retval )) {
      m_logger << kFATAL << " Parameter: " << GetFullName() << " with value: " << retval
               << " out of range: " << GetFitRange() << GEndl;
   }

   return retval;
}

Bool_t Gfitter::GParameter::SetFitValue( Double_t value ) 
{
   Bool_t hasChanged = kTRUE; 

   hasChanged = GetData().SetFitValue( value );   
   if (hasChanged) NotifyReferencesOfChange();

   // if there are links to other ("same") parameters, update these as well
   for (GParPtrVec_t::iterator par = m_samePars.begin(); par != m_samePars.end(); par++) {
      (*par)->SetFitValue( value );
   }

   return hasChanged;
}

Bool_t Gfitter::GParameter::SetValue( Double_t value ) 
{
   Bool_t hasChanged = kTRUE; 
   
   m_data->SetValue (value);
   
   if (hasChanged) NotifyReferencesOfChange();

   // if there are links to other ("same") parameters, update these as well
   for (GParPtrVec_t::iterator par = m_samePars.begin(); par != m_samePars.end(); par++) {
      (*par)->SetValue( value );
   }

   return hasChanged;
}

void Gfitter::GParameter::AddScaler( GScalerBase* scaler, GSclPtrVec_t& scalerVec ) 
{
   // check for duplication
   GSclPtrVec_t::iterator it = scalerVec.begin();
   for (; it != scalerVec.end(); it++) {
      if (**it == *scaler) m_logger << kFATAL << "<AddScaler> Cannot add Scaler \"" 
                                    << scaler->GetParName() << "\" twice" << GEndl;
   }

   scalerVec.push_back( scaler );

   // the presence of scalers changes the fit properties of a parameter that has purely 
   // theoretical ranges
   if (scalerVec.size() == 1) InitFitRanges();
}

const TString Gfitter::GParameter::GetScalerStr( const GSclPtrVec_t& scalerVec ) const
{
   TString sclstr = "";
   for (GSclPtrVec_t::const_iterator it = scalerVec.begin(); it != scalerVec.end(); it++) {
      if (it != scalerVec.begin()) sclstr += ":";
      sclstr += (*it)->GetParName();
   }   

   return sclstr;   
}

void Gfitter::GParameter::InitAllScalers( const GParPtrVec_t& v )
{
   if (HasScalers()) {
      for (GSclPtrVec_t::const_iterator scl = m_scalers.begin(); scl != m_scalers.end(); scl++) {
         GParameter* par = gStore()->GetParameter( (*scl)->GetParName() );
         if (par) (*scl)->SetParameter( par );
      }
   }

   if (HasRescalers()) {
      for (GSclPtrVec_t::const_iterator scl = m_reScalers.begin(); scl != m_reScalers.end(); scl++) {
         for (GParPtrVec_t::const_iterator par = v.begin(); par != v.end(); par++) {
            // the rescaling operations will be done now by rescaling the parameter values and errors
            if (**scl == **par) { (*scl)->SetParameter( *par ); break; }
         }         
      }
   }
}

void Gfitter::GParameter::SetModifyInToy( TString value )
{
   if ( !this->HasTheory() && value != "0" ) {
      m_ModifyInToy = value.Atof();
      m_logger << kINFO << "ModifyInToy is set to " << m_ModifyInToy << GEndl;
   }
   else if ( !this->HasTheory()  ) m_ModifyInToy = 0;
   else if ( this->HasTheory() && value != "0" ) {
      m_logger << kWARNING << "For theory parameters as " << this->GetParName()
               << " : There is no meaning for ModifyInToy!" << GEndl;
      m_ModifyInToy = 0;
   }
}

ostream& Gfitter::operator << ( ostream& os, const GParameter& par ) 
{   
   os << "\"" << par.GetFullName() << "\" : " << par.GetData() 
      << " (Active=" << (par.IsActive() ? "Y" : "N")   
      << "; scan range:" << par.GetScanRange(); 
   if (par.GetPreFitScanRange()) os << "; Pre-fit scan range: " << *par.GetPreFitScanRange();

   if (par.GetTheory() != 0) os << "; theory: "    << *par.GetTheory();
   else                      os << "; no theory";

   return os;
}
