/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GParameter                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Parameter container                                                       *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GParameter
#define Gfitter_GParameter

#include "TString.h"

#include "Gfitter/GObject.h"
#include "Gfitter/GScalerBase.h"
#include "Gfitter/GUtils.h"
#include "Gfitter/GDataBase.h"
#include "Gfitter/GResultBase.h"

class TFormula;

namespace Gfitter {
  
   class GInterval;
   class GScan;
   class GData;
   class GTheory;
   class GParameterRef;
   
   class GParameter;
   std::ostream& operator << ( std::ostream& os, const GParameter& par );

   class GParameter : public GObject {

   public:
    
      // default constructor (needed for persistency - don't use that one)
      GParameter();

      // construct typical measurement or curve
      GParameter( TString parName, TString parAlias, Bool_t active, 
                  GDataBase* data, GInterval* scanRange = 0, TString theoryName = "same" );

      // construct typical theory parameter
      GParameter( TString parName, TString parAlias, Bool_t active, 
                  GInterval* validityRange, GInterval* scanRange = 0, TString theoryName = "same" );

      // copy constructor
      GParameter( const GParameter& );

      // destructor
      virtual ~GParameter();

      // clone function
      void CopyPar( const GParameter& );

      // equality operators: checks ParName only !
      Bool_t operator == ( const GParameter& other ) const { return GetParName() == other.GetParName(); }      
      Bool_t operator != ( const GParameter& other ) const { return GetParName() != other.GetParName(); }      

      // --- accessors
      // parameter name
      const TString&      GetParName()    const { return m_parName; }
      GDataBase&          GetData()       const { return *m_data; }
      GDataBase*          GetDataPtr()    const { return m_data; }

      // refernce data
      GDataBase&          GetRefData()    const { return *m_refdata; }
      GDataBase*          GetRefDataPtr() const { return m_refdata; }
      void                SetRefData( GDataBase* d );

      // alias name (used, eg, for covariance matrix
      // per default: alias name is equal to parameter name
      const TString&      GetParAlias()   const { return m_parAlias; }

      // return the full name: this is ParName_ParAlias if the alias
      // differs from the name, or simply ParName, if ParName==ParAlias
      const TString       GetFullName() const;

      // just as Get(Full)Name (i.e., name without(with) alias), but with 
      // namespace removed (if present)
      const TString       GetParNameWONSpace()  const;
      const TString       GetFullNameWONSpace() const;

      // is alias defined ?
      Bool_t              HasAlias()  const { return m_hasAlias; }
      
      // ---- pure plot functions -----
      // is parameter to be scanned ?
      Bool_t              IsScanned() const { return m_scanned; }

      // has it a data curve or a simple data ?
      Bool_t              IsCurve() const { return m_curve; }

      // range of scan
      GInterval&          GetScanRange()  const { return *m_scanRange; }
      void                SetScanRange( GInterval* s ) { m_scanRange = s; }
      // ------------------------------

      // number of bins used in scans (also for scan histograms)
      Int_t       GetNbins() const        { return m_nbins; }      
      void        SetNbins( Int_t nbins ) { m_nbins = nbins; }

      // is parameter free in fit ?
      // --> see decsription at the inline implementation below
      Bool_t              VariesInFit() const;

      // range of parameter in fit
      // if not set explicitely, use n*ErrGauss + ErrRange
      Double_t            GetFitStep () const { return m_fitStep; }
      void                SetFitStep ( Double_t s ) { m_fitStep = s; }
      GInterval&          GetFitRange() const { return *m_fitRange; }
      void                SetFitRange( GInterval* f );

      // shall the parameter be scanned before fit
      const GInterval*    GetPreFitScanRange() const { return m_preFitScanRange; }
      void                SetPreFitScanRange( GInterval* f );

		// result of actions
      GResultBase*        GetResult()   const { return m_result; }
		void                SetResult( GResultBase* r );

      // is parameter active, does it enter fit ?
      Bool_t              IsActive()      const { return m_active; }
      void                SetActive( Bool_t );

      // ---- theory related ----------
      const TString&      GetTheoryName() const { return m_theoryName; }
      void                SetTheoryName( TString n ) { m_theoryName = n; }

      GTheory*            GetTheory()     const { return m_theory; }
      void                SetTheory( GTheory* t );

      // this provides a cache of persistent results that depend on objects not persistified
      void                UpdatePersistencyCache();
      Double_t            GetCachedCompValue() const { return m_compValCache; }
      Bool_t              GetCachedHasTheory() const { return m_hasTheoryCache; }

      // does associated theory exist ?
      Bool_t              HasTheory()     const;
      // ------------------------------

      // returns -2*log(Likelihood)
      Double_t            GetTheoRef() const;
      Double_t            GetDeviation() const { Int_t sign; return GetDeviation( sign ); }
      Double_t            GetDeviation( Int_t& sign, TFormula* f = 0 ) const;
      Double_t            GetChiSquared() const;

      // returns penalty chi-squared
      Double_t            GetPenaltyChiSquared() const;
    
      // simplified access
      Double_t            GetValue()     const { return GetValue( GetData()    ); }
      Double_t            GetRefValue()  const { return GetValue( GetRefData() ); }
      Bool_t              SetFitValue( Double_t value );
      Bool_t              SetValue( Double_t value ); 
      Double_t            GetFitValue()         const { return GetData().GetFitValue(); }
      Double_t            GetTheoryPrediction() const;

      // backup methods
      GParameter*         GetBackup() const { return m_parBackup; }

      void                Backup();
      void                Recover();

      // ---- caching related ----------
      void                RegisterClientReference( GParameterRef* );
      void                NotifyReferencesOfChange();
      const TList&        GetClientReferenceList() const { return *m_clientReferences; }

      // ---- special treatment to allow parameter duplication
      Bool_t              IsFirst() const { return m_isFirst; }
      void                UnsetFirst() { m_isFirst = kFALSE; }
      const GParPtrVec_t& GetSameParameters() const { return m_samePars; }
      Bool_t              LinkIfSameParameter( GParameter* );

      // ---- for parameter scaling ----
      Bool_t              HasScalers() const { return m_scalers.size() > 0; }
      void                AddScaler( GScalerBase* sc ) { AddScaler( sc, m_scalers ); }
      const GSclPtrVec_t& GetScalers()   const { return m_scalers; }
      const TString       GetScalerStr() const { return GetScalerStr( m_scalers ); }

      Bool_t              HasRescalers() const { return m_reScalers.size() > 0; }
      void                AddRescaler( GScalerBase* sc ) { AddScaler( sc, m_reScalers ); }
      const GSclPtrVec_t& GetRescalers()   const { return m_reScalers; }
      const TString       GetRescalerStr() const { return GetScalerStr( m_reScalers ); }

      void                InitAllScalers( const GParPtrVec_t& );

      // ---- for toy experiments ----
      Double_t            GetGaussRnd ( Double_t mean ) const { return m_data->GetGaussRnd(mean);}
      Double_t            GetRndValue ( Double_t mean ) const { return m_data->GetRndValue(mean); }
      Double_t            GetErrGaussm() const { return m_data->GetErrGaussm(); }  
      Double_t            GetErrGaussp() const { return m_data->GetErrGaussp(); }  

      void                SetModifyInToy( TString value );
      Double_t            GetModifyInToy() const { return m_ModifyInToy; };

   private:

      // members set at construction
      
      TString           m_parName;    // unique parameter identifier
      Bool_t            m_hasAlias;   // is an alias defined ?
      TString           m_parAlias;   // alias name
      GDataBase*        m_data;       //-> the data reference
      GDataBase*        m_refdata;    //-> reference data (used for rescaling)

      Bool_t            m_active;     // is parameter active
      Double_t          m_fitStep;    // size of fit steps
      GInterval*        m_fitRange;   //-> range in fit
      GInterval*        m_scanRange;  //-> scan range       
      Int_t             m_nbins;      // number of bins used in scan
      GInterval*        m_preFitScanRange; //-> pre-fit scan range       

      Double_t          m_ModifyInToy; // Should the parameter modify during toy analysis
                                       // value*sigma, only valid for parameter which are varied
      
      // const parameter to define default fit range
      static const Double_t m_nsig; 

      // members set after construction
      TString           m_theoryName; // name of theory that belongs to the parameter
      GTheory*          m_theory;     //! pointer to the corresponding theory

      // backup of initial settings
      GParameter*       m_parBackup;  //! parameter backup

      Bool_t            m_scanned;    // usually == kFALSE
      Bool_t            m_curve;      // usually == kFALSE

      // private GetValue accessor
      Double_t GetValue( const GDataBase& ) const;

      // ---- caching related ----------
      TList*            m_clientReferences; //! reference to client of this parameter

      // result after actions
      GResultBase*      m_result;           //-> reference to result

      // default initialisation common to all constructors
      // (but the copy constructor)
      void InitTheory( TString theoryName );

      // friend members, which can for example reset the 'scanned' property of a parameter
      friend class GScan;
      friend class GScanAnalysis;

      // this parameter is usually always equal to active; however, in special cases
      // (like scans) this can change, ie, one wants a parameter that has no associated
      // theory to enter the fit, without being free itself
      void SetScanned( Bool_t t ) { m_scanned = t; }

      // initialise fit limits: the limits depend on whether the parameter has 
      // statistical or theory errors
      void InitFitRanges();

      // NOTE: it may occur that several measurements of a same GParameter "A"
      // were added to the datacard. In that case, we should add only once the 
      // parameter to the fit, and the corresponding A::m_fitValues need all to be 
      // set to this single free parameter
      
      // We implement this by adding a flag "IsFirst()" to each parameter. If they 
      // are unique, the flag is "T". If they are non-unique, the flag is only "T"
      // for the first parameter added to the GStore. The parameter will then hold 
      // a list of pointer to the other parameters of the same name, and update their
      // fit values
      Bool_t            m_isFirst;
      GParPtrVec_t      m_samePars;

      // ---- for parameter scaling ----
      GSclPtrVec_t      m_scalers;      //! the scalers are floating parameters
      GSclPtrVec_t      m_reScalers;    //! the *re*scalers correct the GData object only once
      void AddScaler( GScalerBase*, GSclPtrVec_t& );
      const TString GetScalerStr( const GSclPtrVec_t& ) const;

      // this is a cache of the parameter reference value used for persistency
      Double_t          m_compValCache;   // comparison of actual value
      Bool_t            m_hasTheoryCache; // cache for has theory

      ClassDef(GParameter,1) 
   };
}

//
// -------------------- i n l i n e   f u n c t i o n s ---------------------------------
//

inline const TString Gfitter::GParameter::GetParNameWONSpace() const 
{
   return GUtils::RemoveNamespace( GetParName() );
}

inline const TString Gfitter::GParameter::GetFullNameWONSpace() const 
{
   TString retVal = GUtils::RemoveNamespace( GetFullName() ).ReplaceAll("/", "");
   return retVal.ReplaceAll(".", "_");
}

inline const TString Gfitter::GParameter::GetFullName() const 
{
   return (HasAlias() ? m_parName + "." + m_parAlias : m_parName);
}

inline void Gfitter::GParameter::SetResult( GResultBase* result ) 
{ 
   m_result = result; 
   if (m_result) {
      m_result->SetParName( GetFullName() ); 
      m_result->SetRefData( GetDataPtr() );
   }
}

inline Bool_t Gfitter::GParameter::HasTheory() const
{
   return (m_theory != 0);
}

inline void Gfitter::GParameter::SetTheory( GTheory* t ) 
{ 
   m_theory = t; 
}

// is theoretical parameter (as oppposed to measurement)
// ... and is not actually a parameter that is scanned
// (if it were scanned, it should enter the fit, but not be free to vary)

// NOTE: it may occur that several measurements of a same GParameter "A"
// were added to the datacard. If that parameter does not have a theory, 
// we should add only once the parameter to the fit, and the corresponding 
// A::m_fitValues need all to be set to this single free parameter

// We implement this by adding a flag "IsFirst()" to each parameter. If they 
// are unique, the flag is "T". If they are non-unique, the flag is only "T"
// for the first parameter added to the GStore. The parameter will then hold 
// a list of pointers to the other parameters of the same name, and update their
// fit values
inline Bool_t Gfitter::GParameter::VariesInFit() const
{
   return (!HasTheory() && !IsScanned() && IsFirst());
}

#endif
