/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GScalerBase                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Parameter scaler (base class)                                             *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GScalerBase
#define Gfitter_GScalerBase

#include <vector>

#include "Rtypes.h"
#include "GMsgLogger.h"

class TString;
class TFormula;

namespace Gfitter {

   class GParameter;
   class GData;

   // types
   class GScalerBase;
   typedef std::vector<Gfitter::GScalerBase*> GSclPtrVec_t;

   class GScalerBase {

   public:

      GScalerBase( const TString& expression, Double_t errp, Double_t errm, Int_t sign );
      GScalerBase( const GScalerBase& );  
      virtual ~GScalerBase();

      // rescaled central value
      virtual Double_t GetBias() = 0;

      Bool_t operator == (const GScalerBase&    other) const;
      Bool_t operator != (const GScalerBase&    other) const;

      Bool_t operator == (const GParameter& other) const;
      Bool_t operator != (const GParameter& other) const;

      // accessors
      const TString& GetParName()    const { return m_parName; } // this is NOT the name of the mother parameter !!!
      const TString& GetExpression() const { return m_expression; } 
      Double_t       GetErrp()       const { return m_errp; }
      Double_t       GetErrm()       const { return m_errm; }
      Int_t          GetSign()       const { return m_sign; }

      // parameter access (this is the parameter the scaler belongs to (!)
      const GParameter* GetParameter() const { return m_par; }
      virtual void      SetParameter(  const GParameter* par );

      // acess to reference data 
      GData&         GetRefData() const { return *m_refData; }
      void           SetRefValue( Double_t v );

   protected:

      // note that the parameter to which the scaler belongs is not referenced in the 
      // scaler object; it is not necessary

      TString            m_parName;    // this is the name of the parameter that IS the scaler
      TString            m_expression; // this is the expression of the full formula
      Double_t           m_errp;       // these are the effects of the scaler's variations
      Double_t           m_errm;       // these are the effects of the scaler's variations
      Int_t              m_sign;       // the sign of the variations

      const GParameter*  m_par;        // this is the parameter that IS the scaler
      GData*             m_refData;    // reference data (original value of scaler)
      TString            m_refName;    // name of scaler found when interpreting the reference value
      Bool_t             m_first;      // first pass (for precomputation of constants)

      // internal variables used to backup intermediate results
      Double_t           m_denomP;     // denominator for positive error
      Double_t           m_denomM;     // denominator for negative error
      Double_t           m_ref;        // reference offset

      mutable GMsgLogger m_logger;

      Bool_t SanityCheck( const GData* );
   };
}

#endif
