/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GScan                                                                 *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include <iostream>
#include "TAxis.h"
#include "TH1F.h"
#include "TH2F.h"
#include "TTree.h"
#include "TMath.h"
#include "TGraph.h"
#include "Gfitter/GScan.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GUtils.h"
#include "Gfitter/GMinuitFitter.h"
#include "Gfitter/GFitterBase.h"
#include "Gfitter/GTimer.h"
#include "Gfitter/GResultScan1d.h"

using namespace std;
using namespace Gfitter;

Gfitter::GScan::GScan()
{
   m_scanAll = kFALSE;
   InitClassName( "GScan" );
}

Gfitter::GScan::~GScan()
{}

void Gfitter::GScan::Scan1D( const TString& par, Int_t nbins )
{
   GParameter* gpar = gStore()->GetParameter( par );
   if (gpar == 0) {
      m_logger << kFATAL 
               << "<Scan1D> Unknown scan parameter \"" << par << "\": " << " check datacard" << GEndl;
   }     
   gpar->SetNbins(nbins);

   Scan1D( gpar );
}

void Gfitter::GScan::Scan1D( GParameter* gpar )
{
   // sanity check
   if (gpar == 0) m_logger << kFATAL << "<Scan1D> Zero GParameter pointer" << GEndl;

   m_logger << kINFO << "Scanning parameter \"" << gpar->GetParName() << "\""
            << " (full name: \"" << gpar->GetFullName() << "\")"
            << " in interval [" << gpar->GetScanRange().GetMin() << ", " << gpar->GetScanRange().GetMax() << "]"
            << GEndl;
   m_logger << kINFO << "Number of steps in scan: " << gpar->GetNbins() << GEndl;

      
   // book parameter tree
   TTree* FitTree = new TTree( Form( "results_%s", gpar->GetParNameWONSpace().Data() ), 
                               Form( "Results for parameter %s", gpar->GetParNameWONSpace().Data() ) );

   TTree* Forward = new TTree( Form( "forwardresults_%s", gpar->GetParNameWONSpace().Data() ),
                               Form( "Results of forward scan for parameter %s", gpar->GetParNameWONSpace().Data() ) );
      
   // start value tree
   TTree* StartTree = new TTree( Form( "StartVals_%s", gpar->GetParNameWONSpace().Data() ), 
                                 Form( "Start Values for parameter %s", gpar->GetParNameWONSpace().Data() ) );
   

   // how many active parameters
   Int_t num = 0 ;
   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); par1++) num++;

   // create the tree branches
   Double_t x_val[num];
   Double_t x_chi2[num];
   Double_t x_chi2tot;
   Double_t x_StartVal[num];
   Double_t x_StartErrp[num];
   Double_t x_StartErrm[num];
   Int_t i = 0;

   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); par1++) {
      FitTree  ->Branch( (*par1)->GetFullNameWONSpace() , &x_val[i], "x_val/D" );
      FitTree  ->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_chi2" , &x_chi2[i], "x_chi2/D" );
      Forward  ->Branch( (*par1)->GetFullNameWONSpace() , &x_val[i], "x_val/D" );
      Forward  ->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_chi2" , &x_chi2[i], "x_chi2/D" );
      StartTree->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_StartVal" , &x_StartVal[i], "x_StartVal/D" );
      StartTree->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_StartErrp" , &x_StartErrp[i], "x_StartErrp/D" );
      StartTree->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_StartErrm" , &x_StartErrm[i], "x_StartErrm/D" );
      i++;
   }
   FitTree->Branch("chi2_tot" , &x_chi2tot, "x_chi2tot/D" );
   Forward->Branch("chi2_tot" , &x_chi2tot, "x_chi2tot/D" );

   // backup paramter settings
   gpar->Backup();
  
   // clarify for later that this parameter is going to be scanned
   gpar->SetScanned( kTRUE );      
   
   // initialise free parameters in fitter 
   gStore()->GetFitter()->Initialise();
  
   // create histograms (also serves as scan axis)
   TString bulkName = gpar->GetFullNameWONSpace();
   TH1F* th1Chi2  = new TH1F( bulkName + (TString)"_chi2", 
                              bulkName + (TString)" (chi-squared)", 
                              gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1dChi2 = new TH1F( bulkName + (TString)"_dchi2", 
                              bulkName + (TString)" (delta_chi-squared)", 
                              gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1dLike = new TH1F( bulkName + (TString)"_dlike", 
                              bulkName + (TString)" (delta_likelihood)", 
                              gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );
   TH1F* th1CL    = new TH1F( bulkName + (TString)"_cl", 
                              bulkName + (TString)" (1 - CL)", 
                              gpar->GetNbins(), gpar->GetScanRange().GetMin(), gpar->GetScanRange().GetMax() );

   // timing
   Int_t iloop = 0;
   GTimer timer( 2*gpar->GetNbins(), GetPrintName() );

   // two scans (foreward and backwards, to improve fit convergence)
   m_logger << kINFO << "Scanning ... please be patient" << GEndl;
   for (Int_t itype=0; itype<2; ++itype) {

      Int_t ibinStart = (itype == 0) ? 1 : th1Chi2->GetNbinsX();
      Int_t ibinEnd   = (itype == 0) ? th1Chi2->GetNbinsX()+1 : 0;
      Int_t istep     = (itype == 0) ? 1 : -1;
         
      for (Int_t ibin = ibinStart; ibin != ibinEnd; ibin += istep) {
        
         gpar->SetFitValue( th1Chi2->GetBinCenter( ibin ) );
         
         // perform the fit
         Double_t chi2  = gStore()->GetFitter()->ExecuteFit();

         m_logger << kINFO 
                  << Form( "\"%s\" (bin:%4i) value: %8.5g | chi2: %7.4g | time left: %s",
                           gpar->GetFullName().Data(), ibin, gpar->GetFitValue(), chi2, 
                           timer.GetLeftTime( ++iloop ).Data() )
                  << GEndl;
         m_logger << kDEBUG << "=====================================" << GEndl;

         // sanity check: the parameter value (theory) should correspond to the scan point
         if ( !gpar->IsCurve() && gpar->HasTheory() ){ 
            Double_t delta = TMath::Abs(gpar->GetFitValue() -  gpar->GetTheoryPrediction())/gpar->GetFitValue();
            if (delta > 0.01 && gpar->HasTheory()) {
               m_logger << kDEBUG 
                        << "Prediction and Scan-Value for "<< gpar->GetFullName().Data() << " are different" 
                        << "\nDeviation: " << 100*delta << " %"
                        << "\nPrediction: " << gpar->GetTheoryPrediction() << GEndl;
            }  
         }

         // fill the histogram (only if first fill or best chi2)
         if (itype == 0 || chi2 < th1Chi2->GetBinContent( ibin )) th1Chi2->SetBinContent( ibin, chi2 );
	  
         // fill the FitTree 
         Int_t k=0;
         for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
              par1 != gStore()->GetActiveParameters().end(); par1++) {
            if   ( !(*par1)->HasTheory() ) x_val[k]  = (*par1)->GetFitValue();
            else  x_val[k]  = (*par1)->GetTheoryPrediction();
            x_chi2[k] = (*par1)->GetChiSquared();
            x_chi2tot = chi2;
            k++; 
         }
         if (itype == 0) { 
           Forward->Fill();
           Forward->AutoSave();
         }
         if (itype == 1) {
           FitTree->Fill();
           FitTree->AutoSave();
         }
      }
   }

   // fill the StartTree with start values ( mean value and errors )
   Int_t k = 0;
   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); par1++) {
      x_StartVal[k]  =  (*par1)->GetValue();
      x_StartErrp[k] = +(*par1)->GetErrGaussp();
      x_StartErrm[k] = -(*par1)->GetErrGaussm();
      k++;
   }
   StartTree->Fill();

   // correct for minimum-chi2 offset to compute CL
   Float_t chi2min = GUtils::CreateProbTHist( *th1Chi2, *th1dChi2, *th1dLike, *th1CL );

   m_logger << kINFO << "Elapsed time: " << timer.GetElapsedTime() 
            << "                                      " << GEndl;    
   m_logger << kINFO << "Found chi2-min: " << chi2min << GEndl;

   // recover initial state
   gpar->Recover();

   th1Chi2  -> Write();
   th1dChi2 -> Write();
   th1dLike -> Write();
   th1CL    -> Write();

   // store the trees only for a normal 1D scan
   if(!m_scanAll){
      Forward  -> Write();
      FitTree  -> Write();
      StartTree-> Write();
   }
	
   // store the result
	m_result = new GResultScan1d( th1Chi2 );
   
   TTree* Result1dScan = new TTree( "Result_" + gpar->GetFullNameWONSpace() ,
                                    "Result_" + gpar->GetFullNameWONSpace() );
   Double_t res_val, res_chi2, res_err1p, res_err1m, res_err2p, res_err2m, res_err3p, res_err3m;

   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Value" , &res_val, "res_val/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Chi2" , &res_chi2, "res_chi2/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Error1sigp" , &res_err1p, "res_err1p/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Error1sigm" , &res_err1m, "res_err1m/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Error2sigp" , &res_err2p, "res_err2p/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Error2sigm" , &res_err2m, "res_err2m/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Error3sigp" , &res_err3p, "res_err3p/D" );
   Result1dScan->Branch( gpar->GetFullNameWONSpace() + (TString)"_Error3sigm" , &res_err3m, "res_err3m/D" );
   
   res_val   = GetResultScan1d()->GetValue();
   res_chi2  = GetResultScan1d()->GetChi2Min();
   res_err1p = GetResultScan1d()->GetErrGaussAsym(+1);
   res_err1m = GetResultScan1d()->GetErrGaussAsym(-1);
   res_err2p = GetResultScan1d()->GetErrGaussAsym(+2);
   res_err2m = GetResultScan1d()->GetErrGaussAsym(-2);
   res_err3p = GetResultScan1d()->GetErrGaussAsym(+3);
   res_err3m = GetResultScan1d()->GetErrGaussAsym(-3);
   
   Result1dScan->Fill();
   Result1dScan->Write();
   delete Result1dScan;
   delete Forward;
   delete FitTree;
   delete StartTree;
   
   gpar->SetResult( m_result );	
   m_logger << kINFO << "Scan is completed!" << GEndl;
}

void Gfitter::GScan::Scan2D( const TString& par1, const TString& par2, Int_t nbins1,  Int_t nbins2)
{
   GParameter* gpar1 = gStore()->GetParameter( par1 );
   GParameter* gpar2 = gStore()->GetParameter( par2 );

   // sanity checks
   if (gpar1 == 0) m_logger << kFATAL << "<Scan2D> Fatal error: unknown parameter \"" << par1 << "\": "
                            << " check datacard" << GEndl;
   if (gpar2 == 0) m_logger << kFATAL << "<Scan2D> Fatal error: unknown parameter \"" << par2 << "\": "
                            << " check datacard" << GEndl;
   gpar1->SetNbins(nbins1);
   gpar2->SetNbins(nbins2);

   this->Scan2D( gpar1, gpar2 );
}

void Gfitter::GScan::Scan2D( GParameter* gpar1, GParameter* gpar2 )
{
   // sanity check
   if (gpar1 == 0 || gpar2 == 0) m_logger << kFATAL << "<Scan2D> Zero GParameter pointer: " 
                                          << gpar1 << " or " << gpar2 << GEndl;

   m_logger << kINFO << "Scanning parameters \"" << gpar2->GetParName() 
            << "\" versus \"" << gpar1->GetParName() << "\"" 
            << " in the intervals [" << gpar2->GetScanRange().GetMin() << ", " 
            << gpar2->GetScanRange().GetMax() << "] and ["<< gpar1->GetScanRange().GetMin() << ", " 
            << gpar1->GetScanRange().GetMax() << "]"
            << GEndl;
   m_logger << kINFO << "Number of steps in scan for " << gpar1->GetParName() << ": " << gpar1->GetNbins() << GEndl;
   m_logger << kINFO << "Number of steps in scan for " << gpar2->GetParName() << ": " << gpar2->GetNbins() << GEndl;

   // book fit value tree
   TString name1 = Form( "results_%s_vs_%s", 
                         gpar1->GetParNameWONSpace().Data(), gpar2->GetFullNameWONSpace().Data() );
   TString name2 = Form( "Results for parameter scan %s_vs_%s", 
                         gpar1->GetParNameWONSpace().Data(), gpar2->GetFullNameWONSpace().Data() );

   TTree* FitTree = new TTree( name1, name2 );

   // book start value tree
   name1 = Form( "StartVals_%s_vs_%s", 
                         gpar1->GetParNameWONSpace().Data(), gpar2->GetFullNameWONSpace().Data() );
   name2 = Form( "Start Values for parameter scan %s_vs_%s", 
                         gpar1->GetParNameWONSpace().Data(), gpar2->GetFullNameWONSpace().Data() );

   TTree* StartTree = new TTree( name1, name2 );
   
  // how many active parameters
   Int_t num = 0 ;
   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); ++par1) ++num;

   // create the tree branches
   Double_t x_val[num];
   Double_t x_chi2[num];
   Double_t x_chi2tot;
   Double_t x_StartVal[num];
   Double_t x_StartErrp[num];
   Double_t x_StartErrm[num];
   Int_t i = 0;

   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); ++par1) {
      FitTree  ->Branch( (*par1)->GetFullNameWONSpace() , &x_val[i], "x_val/D" );
      FitTree  ->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_chi2" , &x_chi2[i], "x_chi2/D" );
      StartTree->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_StartVal" , &x_StartVal[i], "x_StartVal/D" );
      StartTree->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_StartErrp" , &x_StartErrp[i], "x_StartErrp/D" );
      StartTree->Branch( (*par1)->GetFullNameWONSpace() + (TString)"_StartErrm" , &x_StartErrm[i], "x_StartErrm/D" );
      i++;
   }
   FitTree->Branch("chi2_tot" , &x_chi2tot, "x_chi2tot/D" );

   // backup paramter settings
   gpar1->Backup();
   gpar2->Backup();

   // clarify for later that this parameter is going to be scanned
   gpar1->SetScanned( kTRUE );      
   gpar2->SetScanned( kTRUE );      
   
   // initialise active parameters in fitter 
   gStore()->GetFitter()->Initialise();

   // create histograms (also serves as scan axis)
   TString bulkName = gpar2->GetFullNameWONSpace() + (TString)"_vs_" + gpar1->GetFullNameWONSpace();
   TH2F* th2Chi2  = new TH2F( bulkName + (TString)"_chi2", 
                              bulkName + (TString)" (chi2-squared)", 
                              gpar1->GetNbins(), gpar1->GetScanRange().GetMin(), gpar1->GetScanRange().GetMax(),
                              gpar2->GetNbins(), gpar2->GetScanRange().GetMin(), gpar2->GetScanRange().GetMax() );
   TH2F* th2dChi2 = new TH2F( bulkName + (TString)"_dchi2", 
                              bulkName + (TString)" (delta chi2-squared)", 
                              gpar1->GetNbins(), gpar1->GetScanRange().GetMin(), gpar1->GetScanRange().GetMax(),
                              gpar2->GetNbins(), gpar2->GetScanRange().GetMin(), gpar2->GetScanRange().GetMax() );
   TH2F* th2dLike = new TH2F( bulkName + (TString)"_dlike", 
                              bulkName + (TString)" (delta likelihood)", 
                              gpar1->GetNbins(), gpar1->GetScanRange().GetMin(), gpar1->GetScanRange().GetMax(),
                              gpar2->GetNbins(), gpar2->GetScanRange().GetMin(), gpar2->GetScanRange().GetMax() );
   TH2F* th2CL    = new TH2F( bulkName + (TString)"_cl", 
                              bulkName + (TString)" (1 - CL)", 
                              gpar1->GetNbins(), gpar1->GetScanRange().GetMin(), gpar1->GetScanRange().GetMax(),
                              gpar2->GetNbins(), gpar2->GetScanRange().GetMin(), gpar2->GetScanRange().GetMax() );
      
   // timing
   Int_t iloop = 0;
   GTimer timer( 2*gpar1->GetNbins()*gpar2->GetNbins(), GetPrintName() );
         
   // start the scan
   m_logger << kINFO << "Scanning ... please be patient" << GEndl;
   for (Int_t ibin1=1; ibin1<=th2Chi2->GetNbinsX(); ++ibin1) {

      gpar1->SetFitValue( th2Chi2->GetXaxis()->GetBinCenter( ibin1 ) );

      // two scans (foreward and backwards)
      for (Int_t itype=0; itype<2; itype++) {
            
         Int_t ibinStart = (itype == 0) ? 1 : th2Chi2->GetNbinsY();
         Int_t ibinEnd   = (itype == 0) ? th2Chi2->GetNbinsY()+1 : 0;
         Int_t istep     = (itype == 0) ? 1 : -1;
            
         for (Int_t ibin2 = ibinStart; ibin2 != ibinEnd; ibin2 += istep) {

            gpar2->SetFitValue( th2Chi2->GetYaxis()->GetBinCenter( ibin2 ) );
               
            // perform the fit
            // check if computed chi2 is huge then do nothing
            // no new information is expected
            Double_t chi2 = 0;
            chi2 = gStore()->GetFitter()->ExecuteFit();

            m_logger << kDEBUG << "\"" << gpar2->GetParName() << "\" vs \"" << gpar1->GetParName() << "\""
                     << "\" (bins: " << ibin1 << "/" << ibin2 
                     << ") values: " << gpar1->GetFitValue() << "/" << gpar2->GetFitValue()
                     << " | chi2: " << chi2 << GEndl;
            
            // sanity check: the parameter values (fit-value or theory) should correspond to the scan point
            if( !gpar1->IsCurve() && gpar1->HasTheory() ){
               Double_t delta1 = TMath::Abs(gpar1->GetFitValue() -  gpar1->GetTheoryPrediction())/gpar1->GetFitValue();
               if (delta1 > 0.01 && gpar1->HasTheory()) {
                  m_logger << kDEBUG 
                           << "Prediction and Scan-Value for "<< gpar1->GetParName()
                           << " are different" 
                           << "\nDeviation-par1: " << 100*delta1 << " %" 
                           << ", Scan-Value: " << gpar1->GetFitValue() << GEndl;
               }  
            }
            if( !gpar2->IsCurve() && gpar2->HasTheory() ){
            Double_t delta2 = TMath::Abs(gpar2->GetFitValue() -  gpar2->GetTheoryPrediction())/gpar2->GetFitValue();
            if (delta2 > 0.01 && gpar2->HasTheory()) {
               m_logger << kDEBUG 
                        << "Prediction and Scan-Value for "<< gpar2->GetParName()
                        << " are different" 
                        << "\nDeviation-par2: " << 100*delta2 << " %"
                        << ", Scan-Value: " << gpar2->GetFitValue() << GEndl;
                  } 
            }
            // fill the histogram (only if first fill or best chi2)
            if (itype == 0 || chi2 < th2Chi2->GetBinContent( ibin1, ibin2 )) {
               th2Chi2->SetBinContent( ibin1, ibin2, chi2 );
               Int_t k=0;
               for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
                    par1 != gStore()->GetActiveParameters().end(); ++par1) {
                  if   ( !(*par1)->HasTheory() ) x_val[k]  = (*par1)->GetFitValue();
                  else  x_val[k]  = (*par1)->GetTheoryPrediction();
                  x_chi2[k] = (*par1)->GetChiSquared();
                  x_chi2tot = chi2;
                  k++; 
               }
               FitTree->Fill();
            }
            
            // show progress
            timer.DrawProgressBar( ++iloop );
         }
      }
   }
      
   // fill the StartTree with start values ( mean value and errors )
   Int_t k = 0;
   for (GParPtrVec_t::const_iterator par1 = gStore()->GetActiveParameters().begin(); 
        par1 != gStore()->GetActiveParameters().end(); ++par1) {
      x_StartVal[k]  =  (*par1)->GetValue();
      x_StartErrp[k] = +(*par1)->GetErrGaussp(); 
      x_StartErrm[k] = -(*par1)->GetErrGaussm(); 
      k++;
   }
   StartTree->Fill();

   // correct for minimum-chi2 offset to compute CL
   Float_t chi2min = GUtils::CreateProbTHist( *th2Chi2, *th2dChi2, *th2dLike, *th2CL );

   m_logger << kINFO << "Elapsed time: " << timer.GetElapsedTime() 
            << "                                      " << GEndl;    
   m_logger << kINFO << "Found chi2-min: " << chi2min << GEndl;

   th2Chi2 ->Write();
   th2dChi2->Write();
   th2dLike->Write();
   th2CL   ->Write();
   
   FitTree  -> Write();
   StartTree-> Write();
   
   // recover initial settings
   gpar1->Recover();
   gpar2->Recover();

   delete FitTree;
   delete StartTree;
}

ostream& Gfitter::operator << ( ostream& os, const Gfitter::GScan& /* scan */ ) 
{
   os << 0;
   return os;
}
