/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GSimulatedAnnealing                                                   *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "Gfitter/GSimulatedAnnealing.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GTimer.h"

using namespace std;

Gfitter::GSimulatedAnnealing::GSimulatedAnnealing() 
   : GFitterBase(),
     m_sa(0)
{
   InitClassName( "GSimulatedAnnealing" );
}

Gfitter::GSimulatedAnnealing::~GSimulatedAnnealing()
{
   if (m_sa) delete m_sa;
}

void Gfitter::GSimulatedAnnealing::Initialise()
{
   m_logger << kINFO << "Initialise" << GEndl;
   
   m_This = this;      
   
   // base class call
   GFitterBase::Initialise();  

   this->ReadSAConfiguration();
   this->PrintSAConfiguration();

   // delete TFitter instance if existing
   delete m_sa; m_sa = 0; 

   // reset vectors              
   GetFitPars().clear();
   GetPenalisedPars().clear();

   // define fit parameters      
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();

   for (; par != gStore()->GetActiveParameters().end(); par++) {
      if ((*par)->VariesInFit()) {
         
         // save name and pointer to parameter that is free in the fit
         GetFitPars().push_back( *par );
         
         // sanity check
         if ((*par)->GetFitStep() <= 0) m_logger << "Huge troubles with fit step of par: " 
                                                 << (*par)->GetParName() << GEndl;
         
         // book the fit parameter
         if ( (*par)->GetFitRange().GetMin() ) {
            m_ranges.push_back( new TMVA::Interval( (*par)->GetFitRange().GetMin(),
                                                    (*par)->GetFitRange().GetMax() ) );
         }
         else {
            m_ranges.push_back( new TMVA::Interval( (*par)->GetValue() - 10 * (*par)->GetData().GetErrTotm(),
                                                    (*par)->GetValue() + 10 * (*par)->GetData().GetErrTotm() ) ); 
         }
      }      
   }
  
   m_logger << "Initial parameter ranges: " << GEndl;
   for (UInt_t ipar=0; ipar<m_ranges.size(); ipar++) 
      m_logger << "  ["<< ipar << "] min:" << m_ranges[ipar]->GetMin() << 
         " max:" <<  m_ranges[ipar]->GetMax() << GEndl;
   
   for (par = gStore()->GetParameters().begin(); par != gStore()->GetParameters().end(); par++) {
      if ((*par)->HasTheory() && (*par)->IsScanned() && (*par)->IsFirst()) {
         m_logger << kINFO << "Setting penalised Parameter: " << (*par)->GetParName() << GEndl;
         GetPenalisedPars().push_back( (*par) );
      }
   }        
   
   this->SetNumOfFitPars( m_ranges.size());

   // init covariance matrices and/or diagonal errors
   if (gStore()->IsCorrelationActive()) this->InitErrorMatrix();

   //this->PrintConfiguration();

   m_sa = new TMVA::SimulatedAnnealing( (*(IFitterTarget*)GetThisPtr()) , m_ranges );

   m_sa->SetOptions( m_maxcalls, m_initialtemperature, m_mintemperature, m_eps, m_kerneltemperature,
                  m_temperaturescale, m_adaptivespeed, m_temperatureadaptivestep, 
                  m_usedefaultscale, m_usedefaulttemperature );

}

Double_t Gfitter::GSimulatedAnnealing::GetCurrentChi2()
{
   m_logger << kFATAL << "<GetCurrentChi2> Function not implemented for Simulated Annealing" << GEndl;
   return 0;
}

Double_t Gfitter::GSimulatedAnnealing::ExecuteFit()
{   
   m_logger << kINFO << "ExecuteFit" << GEndl;

   // init timer
   GTimer timer;

   std::vector<Double_t> gvec;
 
   GParPtrVec_t::const_iterator par = gStore()->GetActiveParameters().begin();
   for (; par != gStore()->GetActiveParameters().end(); par++) {
      if ((*par)->VariesInFit())         
         gvec.push_back( (*par)->GetValue() );
   }

   m_logger << "Initial parameter values: " << GEndl;
   for (UInt_t ipar=0; ipar<gvec.size(); ipar++) 
      m_logger << "  ["<< ipar << "] val:" << gvec[ipar] << GEndl;

   m_logger << kINFO << "Start to minimized, please wait ..." << GEndl;
   Double_t fcn = m_sa->Minimize( gvec );
   m_logger << kINFO << "... finished!" << GEndl;

   std::vector<TString>  svec;

   m_logger << kINFO << "Elapsed time: " << GetFitDuration() << GEndl;
   m_logger << kINFO << "SA Results (Minimum = " << fcn << "):" << GEndl;

  // update parameters with these results
   for (UInt_t ipar=0; ipar<GetFitPars().size(); ipar++) {
      GetFitPars()[ipar]->SetFitValue( gvec[ipar] );
      svec.push_back( GetFitPars()[ipar]->GetParName() );
   }

   GUtils::FormattedOutput( gvec, svec, "Parameters", "Best values", m_logger, "%+1.4g" );
   m_logger << GEndl;

   return 0.;
}

void Gfitter::GSimulatedAnnealing::DrawContour( GParameter*, GParameter*, 
                                           Int_t, Double_t, Double_t*, Double_t* )
{
   m_logger << kFATAL << "DrawContour not implemented for GSimulatedAnnealing" << GEndl;
}

void Gfitter::GSimulatedAnnealing::UpdateResults()
{
   m_logger << kFATAL << "UpdateResults not implemented for GSimulatedAnnealing" << GEndl;
}

Gfitter::GSimulatedAnnealing* Gfitter::GSimulatedAnnealing::m_This = 0;

Gfitter::GSimulatedAnnealing* Gfitter::GSimulatedAnnealing::GetThisPtr()
{
   return m_This;
}

Double_t Gfitter::GSimulatedAnnealing::EstimatorFunction( std::vector<Double_t>& parameters )
{ 
   const Int_t npars = parameters.size();
   Double_t    fitPars[npars];
   for (UInt_t i=0; i<parameters.size(); i++) fitPars[i] = parameters[i];
   
   Double_t f = 0.;
   Int_t iflag = 0;
   ((GSimulatedAnnealing*)GetThisPtr())->FCN( f, fitPars, iflag );
   
   return f; 
} 

void Gfitter::GSimulatedAnnealing::ReadSAConfiguration()
{ 

   m_logger << kINFO << "Reading SA configuration" << GEndl;

   // read and set SA parameters
   m_maxcalls = (gStore()->ExistVariable( "SimulatedAnnealing::MaxCalls" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::MaxCalls" )->GetIntValue() : 100000);
   
   m_initialtemperature        = (gStore()->ExistVariable( "SimulatedAnnealing::InitialTemperature" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::InitialTemperature" )->GetFloatValue() : 1e+6);

   m_mintemperature         = (gStore()->ExistVariable( "SimulatedAnnealing::MinTemperature" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::MinTemperature" )->GetFloatValue() : 1e-6);

   m_eps       = (gStore()->ExistVariable( "SimulatedAnnealing::Eps" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::Eps" )->GetFloatValue() : 1e-10);

   m_kerneltemperature      = (gStore()->ExistVariable( "SimulatedAnnealing::KernelTemperature" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::KernelTemperature" )->GetStringValue() : "IncAdaptive");

   m_temperaturescale = (gStore()->ExistVariable( "SimulatedAnnealing::TemperatureScale" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::TemperatureScale" )->GetFloatValue() : 1.0);

   m_adaptivespeed = (gStore()->ExistVariable( "SimulatedAnnealing::AdaptiveSpeed" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::AdaptiveSpeed" )->GetFloatValue() : 1.0);

   m_temperatureadaptivestep = (gStore()->ExistVariable( "SimulatedAnnealing::TemperatureAdaptiveStep" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::TemperatureAdaptiveStep" )->GetFloatValue() : 0.009875);

   m_usedefaultscale = (gStore()->ExistVariable( "SimulatedAnnealing::UseDefaultScale" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::UseDefaultScale" )->GetBoolValue() : kFALSE);

   m_usedefaulttemperature = (gStore()->ExistVariable( "SimulatedAnnealing::UseDefaultTemperature" ) ? 
                       gStore()->GetVariable( "SimulatedAnnealing::UseDefaultTemperature" )->GetBoolValue() : kFALSE);
}

void Gfitter::GSimulatedAnnealing::PrintSAConfiguration()
{ 
   m_logger << kINFO << "SA CONFIGURATION          : " << GEndl;
   m_logger << kINFO << "  MaxCalls                : " << m_maxcalls << GEndl;
   m_logger << kINFO << "  InitialTemperature      : " << m_initialtemperature  << GEndl;
   m_logger << kINFO << "  MinTemperature          : " << m_mintemperature    << GEndl;
   m_logger << kINFO << "  Eps                     : " << m_eps  << GEndl;
   m_logger << kINFO << "  KernelTemperature       : " << m_kerneltemperature << GEndl;
   m_logger << kINFO << "  TemperatureScale        : " << m_temperaturescale << GEndl;
   m_logger << kINFO << "  AdaptiveSpeed           : " << m_adaptivespeed << GEndl;
   m_logger << kINFO << "  TemperatureAdaptiveStep : " << m_temperatureadaptivestep << GEndl;
   m_logger << kINFO << "  UseDefaultScale         : " << m_usedefaultscale << GEndl;
   m_logger << kINFO << "  UseDefaultTemperature   : " << m_usedefaulttemperature  << GEndl;

}
