/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GTheoryBase                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      Abstract base class for theory                                            *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GTheoryBase
#define Gfitter_GTheoryBase

#include <map>
#include "TString.h"
#include "Gfitter/GObject.h"
#include "Gfitter/GParameterRef.h"

namespace Gfitter {

   class GParameter;
   class GTheoryRef;
   class GStore;
   class GTheoryBase;

   std::ostream& operator << ( ostream& os, const GTheoryBase& theory );

   class GTheoryBase : public GObject {
    
   public:
     
      // default constructor
      GTheoryBase();

      // constructor for simple formula parser
      GTheoryBase( const TString& name, const TString& formula );

      // destructor
      virtual ~GTheoryBase();

      /**@brief registering with the parameters that are used by this theory*/
      Bool_t InitParameters();

      /**@brief registering with the theories that are used by this theory*/
      Bool_t InitTheories();

      // derivative of theory prediction used in fit 
      // (does not need to be implemented)
      virtual Double_t GetDerivative()   const { return 0; }
      Bool_t           ExistDerivative() const { return m_existDerivative; }

      // identifier name for theory: determines link to 
      // measurement (parameter)
      const TString&   GetTheoryName() const { return m_theoryName; }

      // instance name: used to distinguish instances of sae theory class
      const TString&   GetInstanceName() const { return m_instanceName; }
      void             SetInstanceName( const TString& i) { m_instanceName = i; }

      // called when attributed to parameter
      virtual void Initialise() = 0;

      // print basic information
      virtual std::ostream& Cout( std::ostream& os, int indLevel = 0, int counter = 1 ) const;

      // ---- caching related ----------
      /**@brief registers a theory as client of this one
       *
       * @param th Theory to be registered as client
       * 
       * A client theory uses the value of this theory for its own
       * calculations
       */
      void RegisterClientReference( GTheoryRef* th );

      /**@brief notifies all client theories of a change*/
      void NotifyReferencesOfChange();
      void BeNotified(GReference & ref);

      Bool_t IsUpToDate() const { return m_isUpToDate; }
      void   SetUpToDate()      { SetReferenceUsed(0); }

   protected:

      void BookParameter( const TString& parname, GParameterRef* par );
      void BookTheory   ( const TString& parname, GTheoryRef* th );

      void SetTheoryName     ( const TString& t ) { m_theoryName = t; SetClassName( t ); }
      void SetExistDerivative( Bool_t e  )        { m_existDerivative = e; }

      void SetReferenceUsed(GReference * parRef);

      Bool_t UseParCaching() const { return m_useParCaching; }

      virtual void UpdateLocalFlags(GReference &) = 0;

      TString m_theoryName;
      TString m_instanceName;
      Bool_t  m_existDerivative;

      // for parameter caching
      Bool_t  m_useParCaching;
      Bool_t  m_isUpToDate;

      std::map<const TString,GParameterRef*> m_TmpParBroker;
      std::map<const TString,GTheoryRef*>    m_TmpTheoryBroker;

      // ---- caching related ----------
      TList m_clientReferences;
      
      ClassDef(GTheoryBase,0)
  };
}

#endif
