/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : MW                                                                    *
 *                                                                                *
 * Description:                                                                   *
 *      Prediction of W mass                                                      *
 *                                                                                *
 * Papers:                                                                        *
 *     (i)  M. Awramik, M. Czakon, A. Freitas and G. Weiglein                     *
 *          Phys. Rev. D 69, 053006 (2004), [hep-ph/0311148]                      *
 *                                                                                *
 *    (ii)  A. Freitas, arXiv:1401.2477                                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
 **********************************************************************************/
#include "TMath.h"
#include "TString.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GConstants.h"

#include "GEW/MW.h"
#include "GEW/DAlphaQED.h"
#include "GEW/RunningAlphaQCD.h"
#include "GEW/AlphaQCDAtQ.h"
#include "GEW/MH.h"

using namespace Gfitter;

ClassImp(GEW::MW)

GEW::MW::MW()
   : Gfitter::GTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );
  
   m_fourLoopCorr = kTRUE ;
   if ( gStore()->ExistVariable( "GEWFlags::FourLoopCorr4Mw" ) ) {
     m_fourLoopCorr = gStore()->GetVariable( "GEWFlags::FourLoopCorr4Mw" )->GetBoolValue();
   }
   m_logger << kINFO << "Using four-loop correction for MW: " << m_fourLoopCorr << GEndl;

   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }

   BookParameter( "MZ",                   &p_MZ );   
   BookParameter( "mt",                   &p_mt );   
   BookParameter( "DeltaMW_Scale",        &p_DeltaMW_Scale );   
   BookParameter( "GF",                   &p_GF );   
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );  
   BookTheory   ( "GEW::RunningAlphaQCD", &t_AlphasRun ); 
   BookTheory   ( "GEW::DAlphaQED",       &t_DAlphaQED );   
   BookTheory   ( "GEW::MH" ,             & t_MH );
}

// init coefficients
// eq. (8) from (i)
void GEW::MW::Initialise()
{
   m_MW0 = 80.3799;
   m_c1  = 0.05429;
   m_c2  = 0.008939;
   m_c3  = 0.0000890;
   m_c4  = 0.000161;
   m_c5  = 1.070;
   m_c6  = 0.5256;
   m_c7  = 0.0678;
   m_c8  = 0.00179;
   m_c9  = 0.0000659;
   m_c10 = 0.0737;
   m_c11 = 114.9;
}

// eq. (6) and (7) from (i)
Double_t GEW::MW::GetTheoryPrediction()
{

   // protection against unphysical values
   if (TMath::Abs(p_MZ)<10e-6 || TMath::Abs(p_MZ)>10e6 ||
       TMath::Abs(p_mt)<10e-6 || TMath::Abs(p_mt)>10e6 || 
       TMath::IsNaN(p_MZ) || TMath::IsNaN(p_mt) ){
     return 0.;
   }

   Double_t MH      = TMath::Abs( GetMH().GetValue() );
   if( m_logMH ) MH = TMath::Exp( TMath::Abs( GetMH().GetValue() ) ); 

   Double_t LogH          = TMath::Log(MH/100);
   Double_t dH          = Gfitter::GMath::IPow( MH/100.0 , 2);

   Double_t dt          = ((p_mt/174.3)*(p_mt/174.3) - 1.0);
   Double_t dZ          = (p_MZ/91.1875) - 1.0;

   Double_t deAlpha     = (GetDAlphaQED().DAlphaQEDMZ()/0.05907) - 1.0;
   Double_t deAlphaS    = (GetAlphasMZ()/0.119) - 1.0;   

   bool print = false;
   if (print){
     std::cout << "MH = " << GetMH().GetValue() << std::endl
	       << "MZ = " << p_MZ << std::endl
	       << "mt = " << p_mt << std::endl
	       << "alphas(MZ) = " << GetAlphasMZ()+0.0 << std::endl
	       << "Dalpha(MZ) = " << GetDAlphaQED().DAlphaQEDMZ() << std::endl;
   }


   Double_t mw = ( m_MW0 - m_c1*LogH - m_c2*LogH*LogH + m_c3*Gfitter::GMath::IPow(LogH,4) 
                   + m_c4*(dH-1) - m_c5*deAlpha 
                   + m_c6*dt - m_c7*dt*dt - m_c8*LogH*dt 
                   + m_c9*dH*dt - m_c10*deAlphaS + m_c11*dZ );

   if (print) std::cout << "MW (old param.) = " << mw << std::endl;
   
   Double_t alphaQEDMZ = GetDAlphaQED().AlphaQEDMZFull();
   Double_t as_mt = GetAlphasRun().EvolveAlphas( p_mt );
   
   // correction due to O(alpha*mt * alphas^3) term as included in (ii)
   // correction to parametrisation formula obtained from A. Freitas, personal correspondence
   Double_t dmw_num = -0.39831 * alphaQEDMZ*alphaQEDMZ * as_mt*as_mt*as_mt * p_mt*p_mt * p_MZ*p_MZ;
   Double_t dmw_den = p_GF * TMath::Power(p_MZ*p_MZ-mw*mw,2) * 
     TMath::Sqrt( 1 - (2*GMath::Sqrt2()*alphaQEDMZ*TMath::Pi()) / (p_GF*p_MZ*p_MZ) ); 
   Double_t dmw = dmw_num / dmw_den;
   Double_t mw_new = mw;
   if ( m_fourLoopCorr ) { mw_new = TMath::Sqrt(mw*mw + dmw); }

   if (print || TMath::IsNaN(dmw_den)){ 
     std::cout << "alphaQED(MZ) = " << alphaQEDMZ << std::endl;
     std::cout << "alphas(mt) = " << as_mt << std::endl;
     std::cout << "squared shift term = " << dmw << std::endl;
     std::cout << "1 = " << TMath::Power(p_MZ*p_MZ-mw*mw,2) << " 2 = " << TMath::Sqrt( 1 - (2*GMath::Sqrt2()*alphaQEDMZ*TMath::Pi()) / (p_GF*p_MZ*p_MZ) )
	       << " 3 = " << (2*GMath::Sqrt2()*alphaQEDMZ*TMath::Pi()) / (p_GF*p_MZ*p_MZ) << std::endl;
     std::cout << "MZ = " << p_MZ << std::endl;
     std::cout << "mw = " << mw << " mw_new = " << mw_new << std::endl;
     std::cout << "MW (with new correction) = " << mw_new << std::endl;
     std::cout << "shift due to O(alpha*mt alphas^3) term = " << mw - mw_new << std::endl;
   }

   // use new value by default
   mw = mw_new;
   
   // theoretical uncertainty in MW
   // eq. (10) from (i)
   mw = mw + (1.0 - p_DeltaMW_Scale)*0.004;

   if (print) std::cout << "MW with theo unc = " << mw << std::endl << std::endl;

   // test GF dependency of GF
   if( false ){
      Double_t MZ2 = p_MZ*p_MZ;

      Double_t deltaMW2 = ( MZ2/2.*( TMath::Sqrt(1-TMath::Sqrt(8.)*TMath::Pi()*GConstants::alphaQED()/(MZ2*GConstants::GF())) 
                                           - TMath::Sqrt(1-TMath::Sqrt(8.)*TMath::Pi()*GConstants::alphaQED()/(MZ2*p_GF)) ) );
      mw = TMath::Sqrt(mw*mw + deltaMW2);                            
   }

   if (TMath::IsNaN(mw)){
     m_logger << kWARNING << "MW is NaN!" << GEndl;
     m_logger << kWARNING << "MH = " << MH << " LogH = " << LogH << " dH = " << dH << " dt = " 
	      << dt << " dZ = " << dZ << " deAlpha = " << deAlpha 
	      << " deAlphaS = " << deAlphaS << GEndl;
     m_logger << kWARNING <<"alphaQED(MZ) = " << alphaQEDMZ << GEndl;
     m_logger << kWARNING <<"alphas(mt) = " << as_mt << " GF = " << p_GF << GEndl;
     m_logger << kWARNING <<"dmw_num = " << dmw_num << " dmw_den = " << dmw_den << GEndl;
     m_logger << kWARNING <<"squared shift term = " << dmw << GEndl;
     m_logger << kWARNING <<"MW (with new correction) = " << mw_new << GEndl;
     m_logger << kWARNING <<"shift due to O(alpha*mt alphas^3) term = " << mw - mw_new << GEndl;
     m_logger << kFATAL << "p_MZ = " << p_MZ << " p_mt = " << p_mt << " MH = " << GetMH().GetValue()
           << " alpha_s = " << GetAlphasMZ()+0.0 << " alpha = " << GetDAlphaQED().DAlphaQEDMZ() << GEndl;
     ///mw = 80.0;
   }
   
   return mw;
}
