/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : R0bParametrisation                                                    *
 *                                                                                *
 * Description:                                                                   *
 *       Ratio of the bottoms partial Z width to all hadrons (full two loop)      *
 *                                                                                *
 * Papers:                                                                        *
 *     (i)  A. Freitas and Yi-Cheng Huang                                         *
 *          arXiv: 1205.0299v1 [hep-ph]                                           *
 *          (need to be confirmed be final publication)                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *     
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GReference.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GConstants.h"

#include "GEW/R0bParametrisation.h"

using namespace Gfitter;

ClassImp(GEW::R0bParametrisation)

GEW::R0bParametrisation::R0bParametrisation() : 
   Gfitter::GAuxTheory()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   const TString& logMH = gStore()->GetVariable( "GEWFlags::logMH" )->GetStringValue();
   m_logger << kINFO << "Using logMH: \"" << logMH << "\"" << GEndl;
   
   if      (logMH == "Yes" ) m_logMH  = kTRUE;
   else if (logMH == "No" )  m_logMH  = kFALSE;
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::logMH\": \"" << logMH << "\""
               << ". Possible are: \"Yes\" and \"No\"\""
               << GEndl;
   }
   
   BookParameter( "MZ",                   &p_MZ );   
   BookParameter( "mt",                   &p_mt );   
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );   
   BookTheory   ( "GEW::DAlphaQED",       &t_DAlphaQED );   
   BookTheory   ( "GEW::MH",              &t_MH );
}

// initialise coefficienct
// eq. (23) of (i) 
void GEW::R0bParametrisation::Initialise()
{
   m_r0b =  0.2154940;
   m_c1  =  1.88e-5;
   m_c2  =  2.0e-6;
   m_c3  = -6.0e-7;
   m_c4  =  8.53e-8;
   m_c5  =  7.05e-4;
   m_c6  = -3.159e-3;
   m_c7  =  6.65e-5;
   m_c8  = -1.704e-3;
   m_c9  = -9.30e-4;
   m_c10 =  6.26e-2;

}


// paramtrized formula of (i), see eqs. (21) and (22)
Double_t GEW::R0bParametrisation::GetR0b()
{   

   Double_t r0b = 0;

   Double_t MH      = TMath::Abs( GetMH().GetValue() );
   if( m_logMH ) MH = TMath::Exp( GetMH().GetValue() ); 
      
   Double_t LogH        = TMath::Log(MH/100.);
   Double_t dH          = MH/100.;

   Double_t dt          = (p_mt/173.2)*(p_mt/173.2) - 1.0;
   Double_t dZ          = (p_MZ/91.1876) - 1.0;

   Double_t deAlpha     = (GetDAlphaQED().DAlphaQEDMZ()/0.05900) - 1.0;
   Double_t deAlphaS    = (GetAlphasMZ()/0.1184) - 1.0;   


   r0b = m_r0b + m_c1*LogH + m_c2*LogH*LogH + m_c3*LogH*LogH*LogH*LogH + m_c4*(dH*dH-1) + m_c5*deAlpha 
     + m_c6*dt + m_c7*dt*LogH + m_c8*deAlphaS + m_c9*deAlphaS*deAlphaS + m_c10*dZ;

   //using namespace std;
   //cout << "R0b: " << " MH = " << MH << " mt = " << p_mt << " MZ = " << p_MZ << " alpha = " << GetDAlphaQED().DAlphaQEDMZ()
   // << " alphaS = " << (deAlphaS + 1.0) * 0.1184 << endl;
   //cout << "R0b = " << r0b << endl << endl;  

   return r0b;
}
