/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : RunningAlphaQCD                                                       *
 *                                                                                *
 * Description:                                                                   *
 *      Auxiliary Theory for running coupling constant                            *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef GEW_RunningAlphaQCD
#define GEW_RunningAlphaQCD

#include "Gfitter/GAuxTheory.h"



namespace GEW {

   class AlphaQCDAtQ;

   class RunningAlphaQCD : public Gfitter::GAuxTheory {

   public:
      
      RunningAlphaQCD();
      ~RunningAlphaQCD() {}

      Double_t GetTheoryPrediction() {return 0.;};

      // init coefficients 
      void Initialise();

      // get alphaQCD at scale mu
      Double_t EvolveAlphas ( Double_t mu );

   protected:
      
      // update of flags in case of parameter change
      void UpdateLocalFlags( Gfitter::GReference& ref );
   
      // update parameters
      void Update();

      // RGE
      Double_t RGE ( Int_t nf, Double_t x, Double_t y );

      // static pointer to this object
      static RunningAlphaQCD* GetThisPointer() { return m_thisPointer; }
    
      // this carrier
      static RunningAlphaQCD* m_thisPointer; 

      // four loop equation for running Alphas
      Double_t AlphaStrong ( Int_t nf, Double_t mu, Double_t lambda ) const;

      // AlphaS matching conditions
      Double_t AlphasMatchDown ( Int_t nl, Double_t as, Double_t mu_nf, Double_t qmass ) const;
      Double_t AlphasMatchUp ( Int_t nl, Double_t as, Double_t mu_nf, Double_t qmass ) const;

      // compute new lambdaMS from Alphas
      Double_t LambdaAtThreshold( Int_t nf, Double_t alphas, Double_t threshold ) const;
      
      // lambdaMS5 for AlphaQCD::EvolveAlphas
      Double_t GetLambdaMS5();

      // static wrapper functions
      static Double_t GetRGE1( Double_t x, Double_t y );
      static Double_t GetRGE2( Double_t x, Double_t y );
      static Double_t GetRGE3( Double_t x, Double_t y );
      static Double_t GetRGE4( Double_t x, Double_t y );
      static Double_t GetRGE5( Double_t x, Double_t y );
      static Double_t GetRGE6( Double_t x, Double_t y );

      //root finding for option FitAlphas
      static Double_t RootAlphas( Double_t lambda );        
      Double_t AlphasAtMZ( Double_t lambda );

      // QCD options
      enum QCDType { kFitLambda = 1, kFitAlphas, kRungeKutta };
      QCDType GetQCDType() const { return m_QCDType; }
      QCDType m_QCDType;

      // parameters from the Data-Card
      Gfitter::GParameterRef p_MZ;
      Gfitter::GParameterRef p_mu;
      Gfitter::GParameterRef p_md;
      Gfitter::GParameterRef p_ms;
      Gfitter::GParameterRef p_mc;
      Gfitter::GParameterRef p_mb;
      Gfitter::GParameterRef p_mt;
      Gfitter::GParameterRef p_Scale;

      // alphasMZ or lambdaMS5
      Gfitter::GParameterRef p_InputValueLambdaMS5;
      Gfitter::GParameterRef p_InputValueAlphasMZ;  

      // fixed order perturbative coefficients
      enum { kNfmax = 6+1 };
      Double_t m_c20;
      Double_t m_c30[kNfmax];
      Double_t m_b0[kNfmax]; 
      Double_t m_b1[kNfmax];
      Double_t m_b2[kNfmax];
      Double_t m_b3[kNfmax];

   private:
    
      // boolean flags to track recomputation
      Bool_t m_isUpToDate_Update; 

      ClassDef(RunningAlphaQCD,0)
   };
}

#endif
