/**********************************************************************************
 * Project: GEW - Electroweak fitting package                                     *
 * Package: GEW                                                                   *
 * Class  : Sigma0had                                                             *
 *                                                                                *
 * Description:                                                                   *
 *      Prediction of hadronic cross-section                                      *
 *                                                                                *
 * see corresponding .h file for author and license information                   *     
 *                                                                                *
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GMath.h"

#include "GEW/Sigma0had.h"
#include "GEW/Z0Base.h"
#include "Gfitter/GVariable.h"

ClassImp(GEW::Sigma0had)

GEW::Sigma0had::Sigma0had() : SMTheoryBuilder()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   BookParameter( "MZ" , &m_MZ );

   // use full two loop calculation, i.e parametrized formula of 
   // paper (i) in Sigma0hadParametrisation

   m_fullTwoLoop = kTRUE;
   if ( gStore()->ExistVariable( "GEWFlags::FullTwoLoop" ) ) {
     m_fullTwoLoop = gStore()->GetVariable( "GEWFlags::FullTwoLoop" )->GetBoolValue();
   }
   m_logger << kINFO << "Using full 2-loop fermionic parametrization for Sigma0had: " << m_fullTwoLoop << GEndl;

   if( m_fullTwoLoop ){
      BookTheory   ( "GEW::Sigma0hadParametrisation",  &t_sig0had );   
   }

   BookParameter( "DeltaAlphasTheoC05_Scale",  &p_DeltaAlphasTheoC05_Scale );
   BookParameter( "DeltaAlphasTheoCMt4_Scale", &p_DeltaAlphasTheoCMt4_Scale );
   BookTheory   ( "GEW::AlphaQCDAtQ/MZ",  &t_AlphasMZ );   

}

Double_t GEW::Sigma0had::GetTheoryPrediction()
{   
   Double_t sigma0had = 0;

   if ( m_fullTwoLoop ) {
     
     sigma0had = GetSigma0hadPara().GetSigma0had();
     
     // and conversion from pb in nb
     sigma0had /= 1000.;

   } else { // using calculation from Z0Parametrisation

     Double_t GaZnu  = GetZ0Theory().GetGaZneutrino();
     
     Double_t GaZe   = GetZ0Theory().GetGaZelectron();
     Double_t GaZmu  = GetZ0Theory().GetGaZmuon();
     Double_t GaZtau = GetZ0Theory().GetGaZtau();
     
     Double_t GaZhad = GetZ0Theory().GetGaZhad();
     
     Double_t GaZtot = 3.0*GaZnu + GaZe + GaZmu + GaZtau + GaZhad;
     
     // and conversion from GeV to nano-barn
     sigma0had = ( 12.0*TMath::Pi()*GaZe*
		   GaZhad/Gfitter::GMath::IPow(GaZtot*m_MZ,2)*
		   Gfitter::GMath::IPow(1.97327,2)*1e5 ); 
   
   }

   // theory uncertainty due to missing h.o. terms in radiator function
   // uncertainty due to C05 term in radiator function
   Double_t asmz = GetAlphasMZ();
   Double_t unc_c05 = -13.0396*GMath::IPow(asmz,4);
   sigma0had += unc_c05*(1 - p_DeltaAlphasTheoC05_Scale);

   // uncertainty due to CMt4 term in radiator function
   Double_t unc_mt4 =  2.11767*GMath::IPow(asmz,4);
   sigma0had += unc_mt4*(1 - p_DeltaAlphasTheoCMt4_Scale);

   return sigma0had;
   
}
