/**********************************************************************************
 * Project: GOblique - Electroweak fitting package                                *
 * Package: GOblique                                                              *
 * Class  : EpsilonParametrisation                                                *
 *                                                                                *
 * Description:                                                                   *
 *      Epsilon Parametrisation                                                   *
 *                                                                                *
 * Sources:                                                                       *
 *      - [ADLO+SLD], Phys. Rept. 427, 257 (2006), [arXiv:hep-ex/0509008]         *
 *      - G. Altarelli, R. Barbieri and F. Caravaglios                            *
 *        Nucl. Phys. B 405, 3 (1993)                                             *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/
#include "TString.h"

#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GConstants.h"
#include "Gfitter/GMath.h"
#include "Gfitter/GTypes.h"
#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"

#include "GOblique/EpsilonParametrisation.h"

#include "TMath.h"

using namespace Gfitter;

ClassImp(GOblique::EpsilonParametrisation)

GOblique::EpsilonParametrisation::EpsilonParametrisation()
       : Gfitter::GAuxTheory(),
         m_isUpToDate_Update( kFALSE )    
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   BookParameter( "MZ",          &p_MZ );   
   BookParameter( "GF",          &p_GF );
   BookParameter( "MH",          &p_MH );
   BookParameter( "mt",          &p_mt );
   
   BookTheory( "GEW::MW",        &t_MW );
   BookTheory( "GEW::DAlphaQED", &t_DAlphaQED );
  
   // SM Predictions
   const TString& theory = gStore()->GetVariable( "GEWFlags::Theory" )->GetStringValue();
   m_logger << kINFO << "Using Theory type: \"" << theory << "\"" << GEndl;
   
   if( theory == "Parametrisation"){
      BookTheory( "GEW::Z0Parametrisation", &t_z0Theory );
   }
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::Theory\": \"" << theory << "\""
               << ". Possible are: \"Parametrisation\"" << GEndl;
   }
}

void GOblique::EpsilonParametrisation::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

void GOblique::EpsilonParametrisation::Update()
{
   if (m_isUpToDate_Update) return;

   // now, it is uptodate (I mean... it will be)
   m_isUpToDate_Update = kTRUE;

   m_alphaMZ = GConstants::alphaQED()/(1 - GetDAlphaQED().DAlphaQEDMZt() );

   // eqs. (4) from Nucl. Phys. B405(1993) 3-23 (note: first eq. and solved to DeltaRho)
   Double_t DeltaRho = -4.0*GetZ0Theory().GetgAeff( GTypes::kElectron, -1/2.0, -1.0 ) - 2.0;
      
   // eq. (7.13) from hep-ex/0509008 
   m_s02      = 0.5*(1.0 - sqrt(1.0 - 4.0 *(TMath::Pi())*m_alphaMZ/( GMath::Sqrt2()*p_GF*p_MZ*p_MZ )));
   m_c02      = 1.0 - m_s02; 
 
   // eqs. (1) from Nucl. Phys. B405(1993) 3-23 (note: solved to Deltarw)
   Double_t Deltarw  = 1.0 - TMath::Pi()*m_alphaMZ / 
      ( GMath::Sqrt2()*p_GF*GetMW()*GetMW()*(1 - GMath::IPow(GetMW()/p_MZ, 2)) );
   
   // eq. (E.1) from hep-ex/0509008 (page 265)
   Double_t sin2Theta = GetZ0Theory().GetSin2Eff(GTypes::kElectron); 
   Double_t Deltak    = sin2Theta/m_s02 - 1.0;

   // eqs. (8) from Nucl. Phys. B405(1993) 3-23 (note: first eq. and solved to DeltaRhob)
   Double_t DeltaRhob = -2.0*GetZ0Theory().GetgAeff( GTypes::kBottom, -1/2.0, -GMath::OneThird() )/
      ( 1.0 + DeltaRho/2.0 ) - 1.0;

   // epsilon parameters
   // eq. (E.2-E.5) from hep-ex/0509008 (page 265)
   m_epsilon1 = DeltaRho;
   m_epsilon2 = m_c02*DeltaRho + m_s02/(m_c02-m_s02)*Deltarw - 2.0*m_s02*Deltak;
   m_epsilon3 = m_c02*DeltaRho + (m_c02-m_s02)*Deltak;
   m_epsilonb = 0.5*DeltaRhob;

   SetUpToDate();
}

// eq. (E.10) from hep-ex/0509008
Double_t GOblique::EpsilonParametrisation::GetS()
{
   Update();
   double s = 4.0*m_s02/m_alphaMZ*(m_epsilon3);
   //if( GetZ0Theory().GetDeltaS() != 0 ) 
   //   s = GetZ0Theory().GetDeltaS()+0.6;

   return s;
}

// eq. (E.11) from hep-ex/0509008
Double_t GOblique::EpsilonParametrisation::GetT()
{
   Update();
   double t = 1.0/m_alphaMZ*(m_epsilon1); 
   //if( GetZ0Theory().GetDeltaT() != 0 ) 
   //   t = GetZ0Theory().GetDeltaT()+0.6;

    return t;
}

// eq. (E.12) from hep-ex/0509008
Double_t GOblique::EpsilonParametrisation::GetU()
{
   Update();

   return 4.0*m_s02/m_alphaMZ*(m_epsilon2);
}

// eq. (E.13) from hep-ex/0509008
Double_t GOblique::EpsilonParametrisation::GetGammab()
{
   Update();

   return 2.29*m_epsilonb;
}

Double_t GOblique::EpsilonParametrisation::GetEpsilon1()
{
   Update();

   return m_epsilon1;
}

Double_t GOblique::EpsilonParametrisation::GetEpsilon2()
{
   Update();

   return m_epsilon2;
}

Double_t GOblique::EpsilonParametrisation::GetEpsilon3()
{
   Update();

   return m_epsilon3;
}

Double_t GOblique::EpsilonParametrisation::GetEpsilonb()
{
   Update();

   return m_epsilonb;
}

