/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : ExtraFamilies                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      STU electroweak corrections of 4th generation                             *
 *      formulae taken from hep-ph/0102144v2                                      *
 *                                                                                *
 * Source:                                                                        *
 *     - H. J. He, N. Polonsky and S. f. Su,                                      *
 *       Phys. Rev. D 64, 053004 (2001), [arXiv:hep-ph/0102144]                   *
 *                                                                                *
 * note: work by DESY summer student Miguel Ariza (Mexico)                        *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"
#include "Gfitter/GMath.h"
#include "Gfitter/GParameter.h"

#include "GSTU/ExtraFamilies.h"

using namespace Gfitter;

ClassImp(GSTU::ExtraFamilies)

GSTU::ExtraFamilies::ExtraFamilies()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   const TString& AbsMass = gStore()->GetVariable( "GSTUFlags::AbsoluteMass" )->GetStringValue();
   m_logger << kINFO << "Use absolute mass " << AbsMass << "\"" << GEndl;

   if(      AbsMass=="Yes" ) m_AbsMass = kTRUE;
   else if( AbsMass=="No"  ) m_AbsMass = kFALSE;
   else{
      m_logger<<kFATAL<<"Unknown option for \"GSTUFlags::AbsoluteMass\": \""<< AbsMass << "\"" 
              << ". Possible are: \"Yes\" or \"No\"" << GEndl;
   }

   const TString& FourthLepton = gStore()->GetVariable( "GSTUFlags::FourthLepton" )->GetStringValue();
   m_logger << kINFO << "Fourth Lepton contribution " << FourthLepton << "\"" << GEndl;
   
   if (FourthLepton=="Enable") m_FourthLepton = kTRUE;
   else if (FourthLepton=="Disable") m_FourthLepton = kFALSE;
   else{
      m_logger<<kFATAL<<"Unknown option for \"GSTUFlags::FourthLepton\": \"" << FourthLepton << "\""
              << ". Possible are: \"Enable\" or \"Disable\""
              << GEndl;
   }
   
   const TString& FourthQuark = gStore()->GetVariable( "GSTUFlags::FourthQuark" )->GetStringValue();
   m_logger << kINFO << "Fourth Quark contribution " << FourthQuark << "\"" << GEndl;
   
   if (FourthQuark=="Enable") m_FourthQuark = kTRUE;
   else if (FourthQuark=="Disable") m_FourthQuark = kFALSE;
   else{
      m_logger<<kFATAL<<"Unknown option for \"GSTUFlags::FourthQuark\": \"" << FourthQuark << "\""
              << ". Possible are: \"Enable\" or \"Disable\""
              << GEndl;
   }

   // book Extra Families parameters
   if (m_FourthLepton ){
      if (m_AbsMass){
         BookParameter( "mLepton4",   &p_mLepton4 );
         BookParameter( "mNeutrino4", &p_mNeutrino4 );
      }
      else{
         BookParameter( "mLepton4",               &p_mLepton4 );
         BookParameter( "mNeutrino4MinusLepton4", &p_mNeutrino4MinusLepton4 );
      }
   }
  
   if (m_FourthQuark){
      if (m_AbsMass){
         BookParameter( "mDown4", &p_mDown4 );
         BookParameter( "mUp4",   &p_mUp4 );
      }
      else{
         BookParameter( "mUp4",           &p_mUp4);
         BookParameter( "mUp4MinusDown4", &p_mUp4MinusDown4);
      }
   }

   // book SM parameters
   BookParameter( "MZ",      &p_MZ );     
   BookTheory   ( "GEW::MW", &t_MW );
}


void GSTU::ExtraFamilies::Initialise(){}


// ----------------------- accessor functions ---------------------------------------

Double_t GSTU::ExtraFamilies::GetS()
{  
   AvoidUnphysicalValues();

   Double_t s = 0;
   
   if( m_FourthLepton ){
      if( m_AbsMass )         
         s += SPar(p_mNeutrino4, p_mLepton4, -0.5, 1.);
      else{
         Double_t mNeutrino4 = p_mLepton4 + p_mNeutrino4MinusLepton4;
         Double_t mLepton4   = p_mLepton4;
       
         s += SPar(mNeutrino4, mLepton4, -0.5, 1.);
      }
   }
   
   if( m_FourthQuark ){
      if( m_AbsMass )
         s += SPar(p_mUp4, p_mDown4, 1./6., 3.);
      else{
         Double_t mUp4   = p_mUp4;
         Double_t mDown4 = p_mUp4 - p_mUp4MinusDown4;
         
         s += SPar(mUp4, mDown4, 1./6., 3.);
      }
   }
   
   return s;
}


Double_t GSTU::ExtraFamilies::GetT()
{
   AvoidUnphysicalValues();

   Double_t t = 0;
  
   if( m_FourthLepton ){
      if( m_AbsMass )
         t += TPar(p_mNeutrino4, p_mLepton4, -0.5, 1.);
      else{
         Double_t mNeutrino4 = p_mLepton4 + p_mNeutrino4MinusLepton4;
         Double_t mLepton4   = p_mLepton4;
                
         t += TPar(mNeutrino4, mLepton4, -0.5, 1.);
      }
   }

   if( m_FourthQuark ){
      if( m_AbsMass )
         t += TPar (p_mUp4, p_mDown4, 1./6., 3.);
      else{
         Double_t mUp4   = p_mUp4;
         Double_t mDown4 = p_mUp4 - p_mUp4MinusDown4;
         
         t += TPar(mUp4, mDown4, 1./6., 3.);
      }
   }
   
   return t;
  
}

Double_t GSTU::ExtraFamilies::GetU()
{
   AvoidUnphysicalValues();
   
   Double_t u = 0;
  
   if( m_FourthLepton ){
      if( m_AbsMass )
         u += UPar(p_mNeutrino4, p_mLepton4, -0.5, 1.);
      else {
         Double_t mNeutrino4 = p_mLepton4 + p_mNeutrino4MinusLepton4;
         Double_t mLepton4   = p_mLepton4;
              
         u += UPar(mNeutrino4, mLepton4, -0.5, 1.);
      }
   }
   
   if( m_FourthQuark ){
      if( m_AbsMass )
         u += UPar(p_mUp4, p_mDown4, 1./6., 3.);
      else{
         Double_t mUp4   = p_mUp4;
         Double_t mDown4 = p_mUp4 - p_mUp4MinusDown4;
         
         u += UPar(mUp4, mDown4, 1./6., 3.);
      }
   }
   return u;
}

// ------------------------------ calculation of STU -------------------------------

// eq. (9) from arXiV:hep-ph/0102144
Double_t GSTU::ExtraFamilies::SPar( Double_t M1, Double_t M2, Double_t Y, Double_t Nc )
{
   Double_t pi=TMath::Pi();
   Double_t x1 = (M1/p_MZ)*(M1/p_MZ);
   Double_t x2 = (M2/p_MZ)*(M2/p_MZ);

   Double_t  spar = ( Nc/(6.*pi)*(2.*(4.*Y+3.)*x1 
                                  + 2.*(-4.*Y+3.)*x2 - 2.*Y*TMath::Log(x1/x2)
                                  + ((3./2.+2.*Y)*x1+Y)*G(x1) + ((3./2.-2.*Y)*x2-Y)*G(x2)) );

   return spar;
}

// eq. (10) from arXiV:hep-ph/0102144
Double_t GSTU::ExtraFamilies::TPar( Double_t M1, Double_t M2, Double_t Y, Double_t Nc )
{
   Double_t MW=Y; MW = GetMW().GetTheoryPrediction(); // MB : quick hack, get rid of compiler warning
   Double_t pi  = TMath::Pi();
   Double_t cw2 = (MW/p_MZ)*(MW/p_MZ);
   Double_t sw2 = 1-(MW/p_MZ)*(MW/p_MZ);
   Double_t x1  = (M1/p_MZ)*(M1/p_MZ);
   Double_t x2  = (M2/p_MZ)*(M2/p_MZ);

   Double_t tpar   = Nc/(8.*pi*sw2*cw2)*F(x1,x2);
   return tpar;
}


// eq. (11) from arXiv::hep-ph/0102144
Double_t GSTU::ExtraFamilies::UPar ( Double_t M1, Double_t M2, Double_t Y, Double_t Nc )
{
   Double_t pi=Y; pi = TMath::Pi(); // MB : quick hack, get rid of compiler warning
   Double_t x1  = (M1/p_MZ)*(M1/p_MZ);
   Double_t x2  = (M2/p_MZ)*(M2/p_MZ);
   Double_t upar;
   
   if(M1==M2)
      upar = 0;
   else{
      upar = ( -Nc/(2.*pi)
               *( (x1+x2)/2.-(x1-x2)*(x1-x2)/3. 
                  + ((x1-x2)*(x1-x2)*(x1-x2)/6.-(x1*x1+x2*x2)/(2.*(x1-x2)))*TMath::Log(x1/x2)
                  + (x1-1.)/6.*f(x1,x1)+(x2-1.)/6.*f(x2,x2)+(1./3.-(x1+x2)/6.-(x1-x2)*(x1-x2)/6.)
                  * f(x1,x2) ) );
   }
   return upar;
}


// ------------------------- auxiliary functions -----------------------------

// eq. (A.9) from arXiv::hep-ph/0102144
Double_t GSTU::ExtraFamilies::F (Double_t x, Double_t y)
{
   Double_t val;
   
   if(x==y)
      val = 0;
   else 
      val = (x+y)/2.-x*y/(x-y)*TMath::Log(x/y);
   return val;
}

// eq. (A.10) from arXiv::hep-ph/0102144
Double_t GSTU::ExtraFamilies::G (Double_t x)
{
   return -4.*TMath::Sqrt(4.*x-1)*TMath::ATan(1/TMath::Sqrt(4.*x-1));
}

// eq. (A.11) from arXiv::hep-ph/0102144
Double_t GSTU::ExtraFamilies::f (Double_t x, Double_t y)
{
   Double_t val = 0;
   Double_t Delta = 2.*(x+y)-(x-y)*(x-y) -1.;
   
   if(Delta == 0)
      val = 0.;
   else if(Delta > 0)
      val = -2.*TMath::Sqrt(Delta)*(TMath::ATan((x-y+1.)/TMath::Sqrt(Delta))-TMath::ATan((x-y-1.)/TMath::Sqrt(Delta)));
   else if (Delta < 0)
      val = TMath::Sqrt(-Delta)*TMath::Log((x+y-1.+TMath::Sqrt(-Delta))/(x+y-1.-TMath::Sqrt(-Delta)));
     return val;
}


// --------------------------- check correctness of values ----------------------

void GSTU::ExtraFamilies::AvoidUnphysicalValues()
{
   if( !m_AbsMass ){
      if( m_FourthLepton ){
         Double_t mNeutrino4 = p_mLepton4 + p_mNeutrino4MinusLepton4;
         Double_t mLepton4   = p_mLepton4;
       
         GParameter *lep4 = gStore()->GetParameter("mLepton4");
         GParameter *diff = gStore()->GetParameter("mNeutrino4MinusLepton4");

         if( mNeutrino4 < 46 ){
            if( lep4->IsActive() && diff->IsScanned() ){
               lep4->SetFitValue( 46 - p_mNeutrino4MinusLepton4 );
            }
            else if( diff->IsActive() && lep4->IsScanned() ){
               diff->SetFitValue( 46 + mLepton4 );
            }
            else{
               m_logger << kFATAL << "<ExtraFamilies> At least one mass below M_Z/2: (mn4, ml4)=(" 
                        << mNeutrino4 << ", " << mLepton4 << ") -->  Please take a look into <ExtraFamilies>!" 
                        << GEndl;
            }
         } 
      }
   
      if( m_FourthQuark ){
         Double_t mUp4   = p_mUp4;
         Double_t mDown4 = p_mUp4 - p_mUp4MinusDown4;
         
         GParameter *up4  = gStore()->GetParameter("mUp4");
         GParameter *diff = gStore()->GetParameter("mUp4MinusDown4");
      
         if( mDown4 < 46 ){
            if( up4->IsActive() && diff->IsScanned() ){
               up4->SetFitValue( 46 - p_mUp4MinusDown4 );
            }
            else if( diff->IsActive() && up4->IsScanned() ){
               diff->SetFitValue( 46 + mUp4 );
            }
            else{
               m_logger << kFATAL << "<ExtraFamilies> At least one mass below M_Z/2: (mu4, md4)=(" 
                        << mUp4 << ", " << mDown4 << ") --> Please take a look into <ExtraFamilies>!" 
                        << GEndl;
            }
         }
      }
   }
}
