/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : SMPredictions                                                         *
 *                                                                                *
 * Description:                                                                   *
 *      SM predictions                                                            *
 *                                                                                *
 * Sources:                                                                       *
 *      - C. P. Burgess, Pramana 45 , S47 (1995), [arXiv:hep-ph/9411257]          *
 *        (see also references)                                                   *
 *      - C. P. Burgess, S. Godfrey, H. Konig, D. London and I. Maksymyk          *
 *        Phys. Rev.  D 49, 6115 (1994), [arXiv:hep-ph/9312291]                   *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/
#include "TMath.h"

#include "Gfitter/GVariable.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GMath.h"

#include "GSTU/SMPredictions.h"

using namespace Gfitter;

ClassImp(GSTU::SMPredictions)

GSTU::SMPredictions::SMPredictions()
   : TheoryHandler()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // use full two loop calculation, i.e parametrized formual of paper (i)
   m_fullTwoLoop4R0b = kTRUE;

   // SM Predictions
   const TString& theory = gStore()->GetVariable( "GEWFlags::Theory" )->GetStringValue();
   m_logger << kINFO << "Using Theory type: \"" << theory << "\"" << GEndl;
   
   if( theory == "Parametrisation"){
      BookTheory( "GEW::Z0Parametrisation", &t_SMz0Theory );
      BookTheory( "GEW::WParametrisation",  &t_SMwTheory );
   }
   else {
      m_logger << kFATAL << "unknown value for \"GEWFlags::Theory\": \"" << theory << "\""
               << ". Possible are: \"Parametrisation\"" << GEndl;
   }
   
   // SM Parameter
   BookParameter( "MZ",   &p_MZ );
   BookParameter( "GF",   &p_GF );
   BookTheory( "GEW::MW", &t_SMMW );
   //BookTheory( "GEW::R0bParametrisation", &t_SMR0b );
}

void GSTU::SMPredictions::UpdateLocalFlags( GReference& /* ref */ )
{
   m_isUpToDate_Update = kFALSE;
}

void GSTU::SMPredictions::Update()
{
   if (m_isUpToDate_Update) return;

   // now, it is uptodate (I mean... it will be)
   m_isUpToDate_Update = kTRUE;
   
   // get SM predictions
   Double_t gammazuu  = GetSM_Z0Theory().GetGaZup();
   Double_t gammazdd  = GetSM_Z0Theory().GetGaZdown();
   Double_t gammazss  = GetSM_Z0Theory().GetGaZstrange();
   Double_t gammazcc  = GetSM_Z0Theory().GetGaZcharm();
   Double_t gammazbb  = GetSM_Z0Theory().GetGaZbottom();
   Double_t gammazlep = GetSM_Z0Theory().GetGaZelectron();
   Double_t gammazmu  = GetSM_Z0Theory().GetGaZmuon();
   Double_t gammaztau = GetSM_Z0Theory().GetGaZtau();
   Double_t gammazneu = GetSM_Z0Theory().GetGaZneutrino();
   Double_t gammazhad = gammazuu + gammazdd + gammazss + gammazcc + gammazbb;
   //Double_t SM_r0b    = gammazbb  / gammazhad;
   m_SMGammaZtot      = gammazhad + gammazlep + gammazmu +gammaztau + 3*gammazneu;
   
   Double_t Ratio_e   = 1.0 - 4.0*GetSM_Z0Theory().GetSin2Eff(Gfitter::GTypes::kElectron);
   Double_t Ratio_c   = ( 1.0 - 4.0*Gfitter::GMath::TwoThird()*
                          GetSM_Z0Theory().GetSin2Eff(Gfitter::GTypes::kCharm) );
   Double_t Ratio_b   = ( 1.0 - 4.0*Gfitter::GMath::OneThird()*
                          GetSM_Z0Theory().GetSin2Eff(Gfitter::GTypes::kBottom) );
   m_SMA0bFB          = 3*( Ratio_e/(1+Ratio_e*Ratio_e)*Ratio_b/(1+Ratio_b*Ratio_b) );
   m_SMA0cFB          = 3*( Ratio_e/(1+Ratio_e*Ratio_e)*Ratio_c/(1+Ratio_c*Ratio_c) );
   m_SMA0lepFB        = 3*( Ratio_e/(1+Ratio_e*Ratio_e)*Ratio_e/(1+Ratio_e*Ratio_e) );
   m_SMAlep           = 2*(Ratio_e)/(1 + Ratio_e*Ratio_e);
   
   // ---------------- W width -----------------
   m_SMGammaWtot = GetSM_WTheory().GetGammaW();
   

   // add radiative STU corrections (table 1 from hep-ph/9411257) 
   // for Z->bb vertex corrections added from eq.(102)-103 of hep-ph/9312291
   m_STUGammaZtot = ( GetSM_GammaZtot()*(1. - 0.707*Get_deltagLb() + 0.128*Get_deltagRb()) 
                          - 0.00961*GetS() + 0.0263*GetT() + 0.0194*GetV() - 0.0207*GetX() );
   gammazhad          = ( gammazhad*(1. - 1.01*Get_deltagLb() + 0.183*Get_deltagRb()) 
                          - 0.00901*GetS() + 0.0200*GetT() + 0.0136*GetV() - 0.0195*GetX() );
   gammazbb           = ( gammazbb*(1. - 4.57*Get_deltagLb() + 0.828*Get_deltagRb()) 
                          -0.00171*GetS() + 0.00416*GetT() + 0.00295*GetV() - 0.00369*GetX() );
   gammazlep         += -0.000192*GetS() + 0.000790*GetT() + 0.000653*GetV() - 0.000416*GetX();
   // gammazcc, coeficients computed from gammazhad and gammazbb
   // up- and down-type quarks havs same contribution
   // cross-checked with "original STU paper" (Phys.Rev.D46:381-409,1992)
   gammazcc          += -0.00194*GetS() + 0.00376*GetT() + 0.002375*GetV() - 0.004215*GetX();

   // compute observables including STU corrections
   m_STUR0b       = gammazbb  / gammazhad; ///  + (m_fullTwoLoop4R0b ? /*GetR0bPara().GetR0b()*/ GetR0bPara().GetTheoryPrediction()  - SM_r0b : 0);
   m_STUR0c       = gammazcc  / gammazhad;
   m_STUR0lep     = gammazhad / gammazlep;
   m_STUSigma0had = 12*TMath::Pi()*gammazlep*gammazhad / 
      TMath::Power(p_MZ*m_STUGammaZtot, 2)*TMath::Power(1.97327,2)*1e5; 

   // set up-to-date
   SetUpToDate();
}

