/**********************************************************************************
 * Project: GSTU - STU Parameter fitting package                                  *
 * Package: GSTU                                                                  *
 * Class  : TechniColor                                                           *
 *                                                                                *
 * Description:                                                                   *
 *      implementation if Technicolor STU Model                                   *
 *                                                                                *
 * Source:                                                                        *
 *      - Ellis, John R. and Fogli, Gian Luigi and Lisi, E., Technicolor          *
 *        and precision electroweak data revisited, Phys.Lett.B343,               * 
 *        282-290,1995, CERN-TH-7448-94, BARI-TH-182-94                           *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/


#include "TMath.h"

#include "Gfitter/GTheory.h"
#include "Gfitter/GTheoryRef.h"
#include "Gfitter/GParameterRef.h"
#include "Gfitter/GConstants.h"

#include "GSTU/TechniColor.h"

using namespace Gfitter;

ClassImp(GSTU::TechniColor)

GSTU::TechniColor::TechniColor()
   : TheoryBase()
{
   SetTheoryName( GetName() );
   SetExistDerivative( kFALSE );

   // book SM parameters
   BookTheory   ( "GEW::MW"  ,&t_MW );
   BookParameter( "MZ"        ,&p_MZ  );
   
   // book RS parameters
   BookParameter( "TechniElectronMass"	, &p_TechniElectronMass ); 
   BookParameter( "TechniLeptonMassRatio"	, &p_TechniLeptonMassRatio ); 
   BookParameter( "NTC"				, &p_NTC   ); 
   BookParameter( "TCWeakHyperCharge"	, &p_TCWeakHyperCharge );
   BookParameter( "DiracMajoranaModel"	, &p_DiracMajoranaModel   );
 
   // -----------------
     
}


void GSTU::TechniColor::Initialise(){}


Double_t GSTU::TechniColor::GetS()
{
	Double_t r		= p_TechniLeptonMassRatio;
	Double_t Y		= p_TCWeakHyperCharge;
	Double_t N_TC	= p_NTC;
	Double_t pi		= TMath::Pi();
	Double_t S		= 0.0;

	/// Dirac Technileptons
	if (p_DiracMajoranaModel<0.5)
	{
		Double_t	S_deg	= 0.0;
		Double_t	S_D	= 0.0;
		// Ellis.et.al: Equation (5)
		S_D	= (N_TC/(6.*pi)) * (1.0-Y*log(r));
		// Ellis.et.al: Equation (8)
		S_deg	= 0.1*(3.0+1.0)*N_TC;
		// Ellis.et.al: Equation (12)
		S	= S_deg + S_D - (N_TC/(6.*pi));
	}
	/// Majorana Technileptons
	else
	{
		// Ellis.et.al: Equation (13)
		Double_t	S_M	= (N_TC/(6.*pi)) * ((-r/pow(1.0+r,2))*(8.0/3.0 + (3.0*r-4.0*r*r + 3.0*r*r*r)/pow((1.0-r*r),2) + 2*log(r)*(pow(r,6)-3.0*pow(r,4)+6.0*pow(r,3)+1)/pow(1.0-r*r,3) ) + ((1.0-r)/(1.0+r))*log(r) + 3.0/2.0 );
		S	= (0.04+0.1*3.0) * N_TC + S_M;
	}
	
	return S;
}


Double_t GSTU::TechniColor::GetT()
{
	Double_t r		= p_TechniLeptonMassRatio;
	Double_t N_TC	= p_NTC;
	Double_t pi		= TMath::Pi();
	Double_t T		= 0.0;

	Double_t M_W	= GetMW().GetTheoryPrediction();
	Double_t M_Z	= p_MZ;


	Double_t cw2	= TMath::Power( M_W/M_Z, 2 );
	Double_t sw2	= 1.0-cw2;

	/// Dirac Technileptons
	if (p_DiracMajoranaModel<0.5)
	{
		// Ellis.et.al: Equation (6)
		T = (N_TC/(16.*pi*cw2*sw2)) * (pow(p_TechniElectronMass,2)/pow(M_Z,2)) * (1.0 + r - 2.0*(r/(r-1.0))*log(r));
	}
	/// Majorana Technileptons
	else
	{
		// Ellis.et.al: Equation (14)
		T = (N_TC/(16.*pi*cw2*sw2)) * (pow(p_TechniElectronMass,2)/pow(M_Z,2)) * (2.0 - log(r)*(4.0*r)/(r*r-1.0) + (4.0*r)/pow(r+1.0,2) * (1.0 - (r*r+1)/(4.0*r) - log(r) * (r*r-r+1)/(r*r-1)) );
	}

	return T;
}

Double_t GSTU::TechniColor::GetU()
{
   Double_t r	= p_TechniLeptonMassRatio;
 	Double_t pi	= TMath::Pi();
	Double_t N_TC	= p_NTC;

	Double_t U	= 0.0;
	/// Dirac Technileptons
	if (p_DiracMajoranaModel<0.5)
	{
		// Ellis.et.al: Equation (7)
		U = (2.0/(6.*pi)) * ((-1.0)*(5.0*r*r - 22.0*r + 5.0)/(3.0*pow(r-1.0,2.0)) + ((r*r*r - 3.0*r*r - 3.0*r + 1.0)/(pow((r-1.0),3)) * log(r)) );
	}
	/// Majorana Technileptons
	else
	{
		// Ellis.et.al: Equation (15)
		U = (N_TC/(6.*pi)) * ( 
	                        (r/(pow(r+1.0,2))) * 
					( 8.0/3.0 + (3.0*r*r*r - 4.0*r*r + 3.0*r)/(pow(r*r-1.0,2)) 
					- 2.0*(pow(r,6)-3.0*pow(r,4)+6.0*r*r*r - 3.0*r*r + 1.0)/(pow(r*r-1.0,3))*log(r))
					+ log(r) * (r*r*r-3*r*r-3*r+1)/(pow(r-1,3))
	                        - 13.0/6.0 + (4.0*r)/(pow(r-1.0, 2)) 
	                        );
	}
	return U;
}
