/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GConstants                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Collection of physical constants                                          *
 *                                                                                *
 * Authors:                                                                       *
 *      Gfitter group                                                             *
 *                                                                                *
 * Redistribution and use in source and binary forms, with or without             *
 * modification, are permitted according to the terms listed in the file          *
 * LICENSE.                                                                       *
 **********************************************************************************/

#ifndef Gfitter_GConstants
#define Gfitter_GConstants

#include "Rtypes.h"
#include "TMath.h"

namespace Gfitter {

   namespace GConstants {

      // lepton masses (unit: GeV)
      Double_t me();
      Double_t mmu();
      Double_t mtau();
      
      // alpha QED at Thompson limit
      Double_t alphaQED();

      // alpha QED at the Z pole mass
      Double_t alphaQEDMZ();

      // G-Fermi (unit: GeV^-2)
      Double_t GF();

      // Bu mass (unit GeV)
      Double_t mBu();

      //K+ mass (unit GeV)
      Double_t mKch();

      // h-bar (in MeV s)
      Double_t hbar();

      // vacuum expectation value, 246 GeV
      Double_t vev();
   }
}

   // lepton masses
inline Double_t Gfitter::GConstants::me()   
{ 
   return 0.000510998918; 
}

inline Double_t Gfitter::GConstants::mmu()  
{ 
   return 0.1056583692; 
}

inline Double_t Gfitter::GConstants::mtau() 
{ 
   return  1.77690; // error: 0.00020 GeV (PDG 2007)
} 

   // alpha QED(s = 0)
inline Double_t Gfitter::GConstants::alphaQED()   
{ 
   return 7.2973525698e-3; // = 1/137.035999074(44) (PDG 2012)
} 

// alpha QED(s = MZ) // the value is wrong here
inline Double_t Gfitter::GConstants::alphaQEDMZ() 
{ 
   return 7.297352568e-3; // = 1/137.03599911(46)
} 

// G-Fermi (unit: GeV^-2)
inline Double_t Gfitter::GConstants::GF() 
{ 
   return 1.16637e-5; // error: 0.00001 GeV^-2 (PDG 2007)
} 

// Bu mass (unit: GeV)
inline Double_t Gfitter::GConstants::mBu() 
{ 
   return 5.27913; // error: 0.00031 GeV (PDG 2007)
} 

// K+ mass (unit: GeV)
inline Double_t Gfitter::GConstants::mKch() 
{ 
  return 0.493677; // +-0.000013 GeV (PDG 2007)
} 

// h-bar (unit: GeV ps)
inline  Double_t Gfitter::GConstants::hbar()
{
   return 6.58211889e-13;
}

// vev (unit: GeV)
inline  Double_t Gfitter::GConstants::vev()
{
   return TMath::Sqrt(1/(TMath::Sqrt(2)*GConstants::GF()));
}

#endif      
