/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GDCardInterpreterXML                                                  *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *         
 *                                                                                *
 **********************************************************************************/

#include "Riostream.h"
#include "TObjString.h"
#include "TDOMParser.h"
#include "TXMLNode.h"
#include "TXMLDocument.h"
#include "TXMLAttr.h"
#include <algorithm> 

#include "Gfitter/GDCardInterpreterXML.h"
#include "Gfitter/GUtils.h"
#include "Gfitter/GMath.h"
#include "Gfitter/GInterval.h"
#include "Gfitter/GData.h"
#include "Gfitter/GDataCurve.h"
#include "Gfitter/GWorkspace.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GStringList.h"
#include "Gfitter/GScalerBase.h"
#include "Gfitter/GAction.h"

using namespace std;

// reference list for allowed attributes of XML nodes
const TString GDCardInterpreterXML_Separator( ":" );
// CanScanned only foe backward compatibility
const TString GDCardInterpreterXML_Attributes_Parameter( "Name:Alias:ModifyInToy:Value:CanScanned:Active:File:Reference:ChiSq:Axes:ForceChi2OfZero:FitLimits:PreFitScan:ScanRange:Rescale:RefValue:WorkspaceFile:WorkspaceName:ModelConfig" );

Gfitter::GDCardInterpreterXML::GDCardInterpreterXML( const TString& xmlFileName )
   : GDCardInterpreterBase( xmlFileName )
{
   InitClassName( "GDCardInterpreterXML" );
}

Gfitter::GDCardInterpreterXML::~GDCardInterpreterXML()
{}

Bool_t Gfitter::GDCardInterpreterXML::Interpret()
{
   // --------------- xml read
   m_logger << kINFO << "Read xml data card: \"" << m_dataCardName << "\"" << GEndl;
  
   TDOMParser* xmlparser = new TDOMParser();
   xmlparser->SetReplaceEntities( kTRUE );
   xmlparser->SetValidate(kFALSE); // dont check dtd file consistency
      
   Int_t parseCode = xmlparser->ParseFile( m_dataCardName );

   m_logger << kINFO << "XML parser returned code: " << parseCode << GEndl;
   if (parseCode != 0) m_logger << kFATAL << "loading of xml document failed" << GEndl;
  
   // --------------- parse JobConfiguration
  
   TXMLDocument* xmldoc = xmlparser->GetXMLDocument();

   TXMLNode* jobConfig_node = xmldoc->GetRootNode();
   TXMLNode* jobConfig_elem = jobConfig_node->GetChildren();
  
   while (jobConfig_elem != 0) {
      if      (jobConfig_elem->GetNodeName() == TString("Data"))         this->ReadData        ( jobConfig_elem );
      else if (jobConfig_elem->GetNodeName() == TString("Correlations")) this->ReadCorrelations( jobConfig_elem );
      else if (jobConfig_elem->GetNodeName() == TString("Actions"))      this->ReadActions     ( jobConfig_elem );
      else                                                               this->ReadAttribs     ( jobConfig_elem );

      // crawl on...
      jobConfig_elem = jobConfig_elem->GetNextNode();
   }

   return kTRUE;
}

void Gfitter::GDCardInterpreterXML::ReadAttribs( TXMLNode* node ) 
{
   if (!node->HasAttributes()) return;
   TListIter attribIt( node->GetAttributes() );
   TXMLAttr* curAttr( 0 );
   while ((curAttr = (TXMLAttr*)attribIt()) != 0) {

      if (m_debug) {
         m_logger << kINFO << node->GetNodeName() << ": " << curAttr->GetName() 
                  << "  =  \"" << curAttr->GetValue() << "\"" << GEndl;
      }

      // add variable
      gStore()->AddVariable( (TString)node->GetNodeName() + "::" + 
                             (TString)curAttr->GetName(), curAttr->GetValue() );
   }
}

void Gfitter::GDCardInterpreterXML::ReadActions( TXMLNode* node ) 
{
   TListIter attribIt( node->GetAttributes() );
   TXMLAttr* curAttr( 0 );
   while ((curAttr = (TXMLAttr*)attribIt()) != 0) {
     
      if (m_debug) {
         m_logger << kINFO << "Actions: " << curAttr->GetName() 
                  << "  =  \"" << curAttr->GetValue() << "\"" << GEndl;
      }
      gStore()->AddAction( new GAction( (TString)curAttr->GetName(), (TString)curAttr->GetValue() ) );  
   }
}

void Gfitter::GDCardInterpreterXML::ReadData( TXMLNode* dataNode ) 
{
   // sanity check
   if (!dataNode->HasChildren()) {
      m_logger << kWARNING << "<Data> ... </Data> Part does not contain any parameters" << GEndl;
      return;
   }

   // retrieve childrens of node
   TXMLNode* node = dataNode->GetChildren();

   std::vector<TString> parameter_names;
   std::vector<TString> parameter_aliases;
   for (; node != 0; node = node->GetNextNode()) {
      if (TString("Parameter") != node->GetNodeName()) continue;

      TXMLAttr* curAttr( 0 );

      // debug messages
      if (m_debug) {
         TListIter attribIt(node->GetAttributes());
         while ((curAttr = (TXMLAttr*)attribIt()) != 0) {
            m_logger << kINFO << node->GetNodeName() << ": " << curAttr->GetName() 
                     << "  =  \"" << curAttr->GetValue() << "\"" << GEndl;
         }
      }

      // create new Parameter
      const TList* attribs = node->GetAttributes();

      // parameter name
      curAttr = (TXMLAttr*)attribs->FindObject( "Name" );
      if (curAttr == 0) m_logger << kFATAL << "No \"Name\" attribute given" << GEndl;

      TString name = curAttr->GetValue();

      // sanity check: make sure all parameter attributes can be understood
      GStringList reflist_Parameter( GDCardInterpreterXML_Attributes_Parameter, GDCardInterpreterXML_Separator );
      if (!CheckList( *attribs, reflist_Parameter )) {
         m_logger << kFATAL << "Problem in \"Parameter\": " << name << GEndl;
      }

      // paramter alias (used primarily for covariance matrices)
      TString alias = name;
      curAttr = (TXMLAttr*)attribs->FindObject( "Alias" );         
      if (curAttr != 0) alias = curAttr->GetValue();


      //check if parameter has already been declared
      std::vector<TString>::iterator it = find( parameter_names.begin(), parameter_names.end(), name);
      std::vector<TString>::iterator it2 = find( parameter_aliases.begin(), parameter_aliases.end(), alias);
      if (it != parameter_names.end() && it2 != parameter_aliases.end()){
	m_logger << kINFO << "Parameter " << name << " already in input. Skip this entry." << GEndl;
	continue;
      }
      parameter_names.push_back(name);
      parameter_aliases.push_back(alias);

      
      // should the parameter modify during Toy analyis by value*sigma
      TString modifyintoy = "0";
      curAttr = (TXMLAttr*)attribs->FindObject( "ModifyInToy" );         
      if (curAttr != 0) modifyintoy = curAttr->GetValue();
     
      // is parameter active ?
      curAttr = (TXMLAttr*)attribs->FindObject( "Active" );
      if (curAttr == 0) m_logger << kFATAL << "No \"Active\" attribute given" << GEndl;

      Bool_t active = (curAttr->GetValue() == TString("T") ? kTRUE : kFALSE);

      GData*       data  = 0;
      GDataCurve*  dataCurve  = 0;
      GWorkspace*  dataWorkspace = 0;
      GSclPtrVec_t scalerVec; scalerVec.clear();

      curAttr = (TXMLAttr*)attribs->FindObject( "Value" );

      if (curAttr != 0) {

         // standard data
         
         TString v = curAttr->GetValue();

         TString valstr  = v;
         TString sclstr  = "";

         // check if there is a variable-dependent systematic error (scalers)
         if (v.Contains("{")) {

            // cut into two parts the "value +- err" and scaling parts
            Int_t ic    = 0;
            Int_t ilast = v.Index('{')-1;
            for (; ilast>3; ilast--) {
               if (v[ilast] == '+' || v[ilast] == '-') ic++;
               if (ic == 2) break;
            }
            GUtils::CutStringInParts( v, ilast, valstr, sclstr );
            // interpret scaling errors; options are:
            // C0: "+- 0.4 {beta, beta=0.5 +- 0.1} +- ..."
            // C1: "+0.4 -0.3 {beta, beta=0.5 +- 0.1} +- ..."
            InterpretScalers( sclstr, scalerVec );
         }

         // is "[a,b]" range given (e.g., theory parameter) ?
         if (valstr.Contains("[")) {
           valstr.ToLower(); 
           if (valstr.Contains("initval "))  valstr.ReplaceAll("initval ","");
           if (valstr.Contains("stepsize ")) valstr.ReplaceAll("stepsize ","");
           TObjArray* ar = valstr.Tokenize(" "); 
           /*if (ar->GetEntries()>2) 
             data = new GData( GInterval( ((TObjString*)ar->At(0))->GetString() ), 
                                          ((TObjString*)ar->At(1))->GetString().Atof(), 
                                          ((TObjString*)ar->At(2))->GetString().Atof() ) ; // range, initial value, stepsize
           else*/ 
           if (ar->GetEntries()>1)
             data = new GData( GInterval( ((TObjString*)ar->At(0))->GetString() ), 
                                          ((TObjString*)ar->At(1))->GetString().Atof() ) ; // range, initial value
	   else
             data = new GData( GInterval( ((TObjString*)ar->At(0))->GetString() ) ) ;      // range only
           delete ar;
         } else if (valstr.Contains("_")) {
            // new setting, which should become default
            // D0: [x1:x2]
            // D1: x +- dx_stat +- dx1_syst +- dx2_syst ... +- dt1_theo +- ... +- ds1 (scaler1) +- ds2 (scaler2) +- ...
            // D2: also allowed: +- dx_statsyst
            // D3: same as D1 with any type of asymmetric errors
            Double_t val;
            vector<Double_t> estatp;
            vector<Double_t> estatm;
            vector<Double_t> esystp;
            vector<Double_t> esystm;
            vector<Double_t> etheop;
            vector<Double_t> etheom;
            GUtils::InterpretValueErrorsString( valstr, val, estatp, estatm, esystp, esystm, etheop, etheom, m_logger );

            // create final errors
            vector<Double_t>::iterator it;

            // statistical and systematic errors are Gaussian
            Double_t err_gp = 0, err_gm = 0;
            for (it = estatp.begin(); it != estatp.end(); it++) err_gp += (*it)*(*it);
            for (it = estatm.begin(); it != estatm.end(); it++) err_gm += (*it)*(*it);
            for (it = esystp.begin(); it != esystp.end(); it++) err_gp += (*it)*(*it);
            for (it = esystm.begin(); it != esystm.end(); it++) err_gm += (*it)*(*it);
            err_gp = sqrt(err_gp);
            err_gm = sqrt(err_gm);

            // theoretical errors are ranges
            Double_t err_rp = 0, err_rm = 0;
            for (it = etheop.begin(); it != etheop.end(); it++) err_rp += (*it);
            for (it = etheom.begin(); it != etheom.end(); it++) err_rm += (*it);
            
            // create a GData object which later is added to the GParameter (still to be created)
            data = new GData( val, err_gp, err_gm, err_rp, err_rm );
         }
         else {
            // interpret value and errors; options are:
            // C0: [x1:x2]
            // C1: x +- dx_gauss
            // C2: x +- dx_gauss +- dx_range
            // C3: x + dxp_gauss -  dxm_gauss
            // C4: x + dxp_gauss -  dxm_gauss +- dx_range
            // C5: x +- dx_gauss +  dxp_range -  dxm_range
            // C6: x + dxp_gauss -  dxm_gauss +  dxp_range - dxm_range
            
            // value and errors given
            const Int_t  n = (Int_t)valstr.Length();
	    
            const Double_t none = GMath::None();
            const TString  M = "-";
            const TString  P = "+";
            Double_t val = none;
            Double_t err_gp = none, err_gm = none;
            Double_t err_rp = none, err_rm = none;
            Bool_t   sym = kFALSE;
	    
            // scan input string
            TString label;
            label.Append(valstr(0));
            for (Int_t i=1; i<n; i++) {
               label.Append(valstr(i));
               if (valstr(i-1) != (TString)"e" && valstr(i-1) != (TString)"E") {
                  if (valstr(i) == P) {
                     label.Chop();
                     if      (val    == none) val = label.Atof();
                     else if (err_gm == none) {
                        err_gm = TMath::Abs(label.Atof());
                        if (sym) { err_gp = err_gm; sym = kFALSE; }
                     }
                     else err_rm = label.Atof();
                     label.Resize(0);
                  }
                  if (valstr(i) == M && valstr(i-1) == P) sym = kTRUE;               
                  if (valstr(i) == M && !sym) {
                     label.Chop();
                     if (err_gp == none) err_gp = label.Atof();
                     else                err_rp = label.Atof();
                     label.Resize(0);
                  }
               }
            }
            if (err_gm == none) {
               err_gm = TMath::Abs(label.Atof());
               if (sym) err_gp = err_gm;
            }
            else if (err_rm == none) {
               err_rm = TMath::Abs(label.Atof());
               if (sym) err_rp = err_rm;
            }

            // replace "none" flags
            if (err_gp == none) err_gp = 0;
            if (err_gm == none) err_gm = 0;
            if (err_rp == none) err_rp = 0;
            if (err_rm == none) err_rm = 0;
            
            // sanity check
            if (err_gp == 0 && err_rp == 0) {
               m_logger << kFATAL << "Problem in positive error for parameter: \"" << name << "\""
                        << " and value-string: " << valstr << GEndl;
            }
            if (err_gm == 0 && err_rm == 0) {
               m_logger << kFATAL << "Problem in negative error for parameter: \"" << name << "\""
                        << " and value-string: " << valstr << GEndl;
            }
            
            // create a GData object which later is added to the GParameter (still to be created)
            data = new GData( val, err_gp, err_gm, err_rp, err_rm );
         }
      }
      else if ((curAttr = (TXMLAttr*)attribs->FindObject( "File" )) != 0) {
	
         // parameter with DataCurve
         TString fileName   = curAttr->GetValue();
	 
         curAttr = (TXMLAttr*)attribs->FindObject( "Reference" );
         if (curAttr == 0) m_logger << kFATAL << "No \"Reference\" attribute given" << GEndl;
         TString histName   = curAttr->GetValue();

         curAttr = (TXMLAttr*)attribs->FindObject( "ChiSq" );
         if (curAttr == 0) m_logger << kFATAL << "No \"ChiSq\" attribute given" << GEndl;
         TString formula   = curAttr->GetValue();

         curAttr = (TXMLAttr*)attribs->FindObject( "Axes" );
         if (curAttr == 0) {
             m_logger << kINFO << "Assuming 1D distribution" << GEndl;
             dataCurve = new GDataCurve(fileName, histName, formula);	 
         } else {
             m_logger << kINFO << "Assuming 2D distribution" << GEndl;
             TString axisNames = curAttr->GetValue();
             // check if there is axis definition, if not pickup right axis according to name
             GStringList axes_list(axisNames);
             if (axes_list[0] != "x" || axes_list[0] != "X" || axes_list[0] != "y" || axes_list[0] != "Y" ){
                 if      (axes_list[0] == name) axisNames = "x:"+axisNames;
                 else if (axes_list[1] == name) axisNames = "y:"+axisNames;
                 else m_logger<<kFATAL<<"Wrong format in Axes definition: No axis name corresponds to parameter \""  << name << "\"" << GEndl;
             }
             m_logger << kINFO << "axis names: \"" << axisNames << "\"" << GEndl;

             /// Check if this needs to be a dummy chi2.
	     curAttr = (TXMLAttr*)attribs->FindObject( "ForceChi2OfZero" );
             Bool_t forceChi2OfZero(false);
             if (curAttr != 0) {
                 forceChi2OfZero = (curAttr->GetValue() == TString("T") ? kTRUE : kFALSE);
             }
             dataCurve = new GDataCurve(fileName, histName, formula, axisNames, forceChi2OfZero);	 
         }

      } 
      else if ((curAttr = (TXMLAttr*)attribs->FindObject( "WorkspaceFile" )) != 0) {
	
         // parameter with DataCurve
         TString fileName   = curAttr->GetValue();
	 
         curAttr = (TXMLAttr*)attribs->FindObject( "WorkspaceName" );
         if (curAttr == 0) m_logger << kFATAL << "No \"WorkspaceName\" attribute given" << GEndl;
         TString workspaceName   = curAttr->GetValue();

         curAttr = (TXMLAttr*)attribs->FindObject( "ModelConfig" );
         if (curAttr == 0) m_logger << kFATAL << "No \"ModelConfig\" attribute given" << GEndl;
         TString modelconfigstr   = curAttr->GetValue();

         dataWorkspace = new GWorkspace(fileName, workspaceName, modelconfigstr);	 
      } 
      else {
         m_logger << kFATAL << "Error with parameter '" << name << "' neither Data nor DataCurve nor Workspace available!" << GEndl;
      }
      
      // scan range
      TXMLAttr*  srAtt     = (TXMLAttr*)attribs->FindObject("ScanRange");
      GInterval* scanRange = 0;
      if (srAtt != 0) scanRange = new GInterval( srAtt->GetValue() );
		
      // create the parameter
      GParameter* par=0;
      if      (data)          par = new GParameter( name, alias, active, data         , scanRange );
      else if (dataCurve)     par = new GParameter( name, alias, active, dataCurve    , scanRange );
      else if (dataWorkspace) par = new GParameter( name, alias, active, dataWorkspace, 0         );
      else {
         m_logger << kFATAL << "Cannot create parameter \"" << name 
                  << "\" neither Data nor DataCurve nor Workspace available!" << GEndl;
      }
    
      // add scalers for this parameter ?
      if (scalerVec.size() > 0) {
         for (GSclPtrVec_t::const_iterator it = scalerVec.begin(); it != scalerVec.end(); it++) {
            par->AddScaler( *it );
         }
      }         

      // are fit limits given ?
      curAttr = (TXMLAttr*)attribs->FindObject( "FitLimits" );    
      GInterval* fitRange = 0;
      if (curAttr != 0){ 
			fitRange = new GInterval( curAttr->GetValue() );
			m_logger << kINFO << "Overwriting fit range of parameter " << name << " to [" << fitRange->GetMin() << "," << fitRange->GetMax() << "]" << GEndl;
			par->SetFitRange( fitRange );
		}

      // is a pre-fit scan range given ?
      curAttr = (TXMLAttr*)attribs->FindObject( "PreFitScan" );         
      if (curAttr != 0) par->SetPreFitScanRange( new GInterval( curAttr->GetValue() ) );
    
      // is "Rescale" information given ?
      curAttr = (TXMLAttr*)attribs->FindObject( "Rescale" );         
      if (curAttr != 0) {
         // interpret scaling errors; options are:
         // C0: "+- 0.4 (beta) +- ..."
         // C1: "+0.4 -0.3 (beta) +- ..."
         scalerVec.clear();
         TString sclstr = curAttr->GetValue();
         InterpretScalers( sclstr, scalerVec );
         for (GSclPtrVec_t::const_iterator it = scalerVec.begin(); it != scalerVec.end(); it++) {
            par->AddRescaler( *it );
         }         
      }

      // Reference values (those are original values, overwritten by the current "value"
      curAttr = (TXMLAttr*)attribs->FindObject( "RefValue" );

      if (curAttr !=0 ) {

         TString v = curAttr->GetValue();

         // options:
         // C1: x +- dx_gauss
         // C2: x + dxp_gauss -  dxm_gauss

         Double_t val = 0, errp = 0, errm = 0;
         GUtils::InterpretValueErrorString( v, val, errp, errm, m_logger ); 
         
         par->SetRefData( new GData( val, errp, errm ) );
      }

      // some debugging output
      m_logger << kINFO << "Create GParameter: \"" << par->GetFullName() << "\""
               << (par->HasScalers()   ? Form( " (contains scalers:   \"%s\")", par->GetScalerStr().Data() )   : "")
               << (par->HasRescalers() ? Form( " (contains REscalers: \"%s\")", par->GetRescalerStr().Data() ) : "")
               << GEndl;
      gStore()->AddParameter( par ); // the Store owns the parameter now
      
      // Set ModifyInToy
      par->SetModifyInToy( modifyintoy );

      m_logger << GEndl; 
   }
}

void Gfitter::GDCardInterpreterXML::ReadCorrelations( TXMLNode* corrNode ) 
{
   if (!corrNode->HasAttributes()) {
      m_logger << kWARNING << "<Correlations> ... </Correlations> Part does not contain any attributes" << GEndl;
      return;
   }

   // lists for parameters and coefficeints
   GStringList* parList  ( 0 );
   GStringList* coeffList( 0 );

   TListIter attribIt( corrNode->GetAttributes() );
   TXMLAttr* curAttr( 0 );
   while ((curAttr = (TXMLAttr*)attribIt()) != 0) {

      if (m_debug) {
         m_logger << kINFO << corrNode->GetNodeName() << ": " << curAttr->GetName() 
                  << "  =  \"" << curAttr->GetValue() << "\"" << GEndl;
      }

      TString theName = (TString)curAttr->GetName();
         
      if (theName == "Active")             // active flag ?
         gStore()->SetIsCorrelationsActive( (TString)curAttr->GetValue() == "T" ? kTRUE : kFALSE );         
      else if (theName == "ParAliases")    // read parAlias string
         parList   = new GStringList( (TString)curAttr->GetValue() );         
      else if (theName == "Coefficients")  // retrieve correlation coefficients
         coeffList = new GStringList( (TString)curAttr->GetValue() );
   }

   // only continue interpretation if correlations set active
   if (gStore()->IsCorrelationActive()) {

      m_logger << kINFO << "Correlations activated: store correlation matrix" << GEndl;

      // sanity check
      if (parList == 0 || coeffList == 0) {
         m_logger << kFATAL << "<ReadCorrelations> Correlation matrix is active, but "
                  << "no \"ParAliases\" or \"Coefficients\" are given: " 
                  << parList << " | " << coeffList << GEndl;
      }
         
      Int_t n = parList->GetEntries();
         
      // sanity check - give help message
      if (coeffList->GetEntries() != (n*n + n)/2) {
         m_logger << kINFO << "Format (example - space between ':' and letters are allowed): " << GEndl;
         m_logger << kINFO << "ParAliases    = \"par1 : par2 : par3 : par4\"" << GEndl;
         m_logger << kINFO << "Coefficiencts = \"1    : 0.50 : 0.20 : 0.10 :" << GEndl;
         m_logger << kINFO << "                        1    : 0.10 : 0.50 :"  << GEndl;
         m_logger << kINFO << "                               1    : 0.20 :"  << GEndl;
         m_logger << kINFO << "                                      1\""     << GEndl;
         m_logger << kFATAL << "<ReadCorrelations> Fatal error during reading of "
                  << "correlation matrix: mismatch in number of parameters and/or number of "
                  << "coefficients: " << coeffList->GetEntries() << " | " << Int_t((n*n - n)/2) << GEndl;
      }
         
      // fill GStore (only non-diagonal elements)
      Int_t ic( 0 );
      for (Int_t iv=0; iv<n; iv++) {
         for (Int_t jv=iv; jv<n; jv++, ic++) {
            if (iv != jv) {
               gStore()->AddCorrelation( (*parList)[iv], (*parList)[jv], (*coeffList)[ic].Atof() );
            }
         }
      }            
   }

   delete parList;
   delete coeffList;

   // debug output
   if (m_debug) {

      GUtils::PrintParVector( m_logger, gStore()->GetActiveParameters(),             "Active parameters" );
      GUtils::PrintParVector( m_logger, gStore()->GetActiveCorrelatedParameters(),   "Correlated parameters" );
      GUtils::PrintParVector( m_logger, gStore()->GetActiveUncorrelatedParameters(), "Uncorrelated parameters" );
         
      // sanity check
      if (gStore()->GetActiveParameters().size() != 
          gStore()->GetActiveCorrelatedParameters().size() + gStore()->GetActiveUncorrelatedParameters().size()) {
         m_logger << kFATAL << "<ReadCorrelations> Wrong number of parameters: "
                  << gStore()->GetActiveParameters().size() << " | " 
                  << gStore()->GetActiveCorrelatedParameters().size() << " | " 
                  << gStore()->GetActiveUncorrelatedParameters().size() << GEndl;
      }
   }      
} 

