/**********************************************************************************
 * Project: Gfitter - A ROOT-integrated generic fitting package                   *
 * Package: Gfitter                                                               *
 * Class  : GDataCurve                                                            *
 *                                                                                *
 * Description:                                                                   *
 *      Implementation                                                            *
 *                                                                                *
 * see corresponding .h file for author and license information                   *
 *                                                                                *         
 **********************************************************************************/

#include <cmath>
#include <iostream>

#include "TROOT.h"
#include "TFile.h"
#include "TFormula.h"
#include "TF1.h"
#include "TH2F.h"
#include "TMVA/TSpline1.h"
#include "TMVA/TSpline2.h"

#include "Gfitter/GDataCurve.h"
#include "Gfitter/GStringList.h"
#include "Gfitter/GStore.h"
#include "Gfitter/GParameter.h"
#include "Gfitter/GGraph2PDF.h"
#include "Gfitter/GMsgLogger.h"

using namespace std;

ClassImp(Gfitter::GDataCurve)

Gfitter::GDataCurve::GDataCurve()
   : GDataBase(),
     m_chi2hist( 0 ),
     m_graph   ( 0 ),
     m_offset  ( 0 ),
     m_forceChi2OfZero( false )
{
   InitClassName( "GDataCurve" );
}

Gfitter::GDataCurve::GDataCurve( TString fileName, TString histName, TString formula, TString axisNames, Bool_t forceChi2OfZero )
   : GDataBase()
{
   InitClassName( "GDataCurve" );

   m_forceChi2OfZero = forceChi2OfZero;

   bool flip_axis = false;
   bool is_2D = (!axisNames.IsNull());
   if(is_2D){
       GStringList axes_list(axisNames);
       //check whether flip axis
       if      (axes_list[0] == "x" || axes_list[0] == "X") flip_axis=false;
       else if (axes_list[0] == "y" || axes_list[0] == "Y") flip_axis=true;
       else {
           m_logger << kFATAL
                    << "Wrong format in Axes definition of file \"" << fileName 
                    << "\": missing the axis of parameter in first field; args: \""  << axisNames << GEndl;
       }
       if(flip_axis){
           m_parName       = axes_list[2];
           m_parNameFriend = axes_list[1];
       } else {
           m_parName       = axes_list[1];
           m_parNameFriend = axes_list[2];
       }
   }

   m_logger << kINFO << "Initializing a GDataCurve" << GEndl;
   m_logger << kINFO << "Root File: \"" << fileName 
            << "\" Histogram: \"" << histName << "\" Formula: \"" << formula << "\"" << GEndl;
   if(is_2D){
       m_logger << kINFO << " X-name: \"" <<  m_parName << "\" Y-name: \"" << m_parNameFriend << "\"" << GEndl;
       m_logger << kINFO << " Forcing chi2 to zero: " << m_forceChi2OfZero << GEndl;
   }

   // open root file
   TFile *f1 = new TFile(fileName);
   if (f1==0) m_logger << kFATAL << "Error opening file" << fileName << GEndl;     
   TH1 *hist = 0;
   if(is_2D && flip_axis){
       // retrieve flipped histo
       hist = flip_hist_axis(histName,f1); 
   } else {
       // retrieve original histo
       hist = (TH1*)f1->Get(histName); 
   }
   if (hist==0) {
      m_logger << kFATAL << "Error histogram \"" << histName << "\" in \"" << fileName << "\"" << GEndl;     
   }
   gROOT->cd();

   // transform histo to chisq2 histo using formula
   TFormula* tFormula= new TFormula( "converting_formula", formula.Data() );
   m_chi2hist = (TH1*) hist->Clone(); 
   Int_t Nbins = ( is_2D ? (hist->GetNbinsX()+2) * (hist->GetNbinsX()+2) : (hist->GetNbinsX()+2) );
   for (int ibin=0;ibin<Nbins; ibin++) {
      m_chi2hist->SetBinContent(ibin, tFormula->Eval(hist->GetBinContent(ibin)));
   }
   delete tFormula;

   // close file
   f1->Close();

   // get graph from histo
   if(!is_2D){
       m_graph = new TGraph( m_chi2hist );
       m_spline = new TMVA::TSpline1( "m_spline", m_graph );
   } else {
       m_graph = new TGraph(m_chi2hist->GetNbinsX());
   }

   // initialisation
   m_fitValue  = GetValue();
}

Gfitter::GDataCurve::GDataCurve( const GDataCurve& dataCurve )
   : GDataBase( dataCurve ),
     m_chi2hist( 0 ),
     m_graph   ( 0 ),
     m_offset  ( 0 )
{
   InitClassName( "GDataCurve" );

   TString hname(dataCurve.m_chi2hist->GetName());
   m_chi2hist = (TH1*) dataCurve.GetHist().Clone("hname");
   m_graph    = new TGraph( dataCurve.GetGraph() );
   if(m_chi2hist->GetDimension()==1){
       m_spline   = new TMVA::TSpline1( "m_spline", m_graph );
   } else {
       m_parName = dataCurve.m_parName;
       m_parNameFriend = dataCurve.m_parNameFriend;
   }
}

Gfitter::GDataCurve::~GDataCurve()
{
   if (m_graph)    delete m_graph;
   if (m_chi2hist) delete m_chi2hist;
   //if (m_spline)   delete m_spline;
}

Double_t Gfitter::GDataCurve::GetChiSquared( Double_t fitValue ) const
{
  Double_t retVal=0.;

  if (m_forceChi2OfZero) { return 0.0; }

  if ( (fitValue > this->GetMax()) || (m_fitValue < this->GetMin()) ) {
      m_logger << kDEBUG << "Out of range x=" << fitValue << GEndl;
      retVal = 1000;
  } else {

      if (m_chi2hist->GetDimension()==1){
          retVal = m_chi2hist->Interpolate(fitValue) ;
          //retVal = m_spline->Eval(  fitValue - m_offset );
      } else if(m_chi2hist->GetDimension()==2){
          GParameter *parF =  gStore()->GetParameter(m_parNameFriend);
          Double_t y = parF->HasTheory() ? parF->GetTheoryPrediction() : parF->GetFitValue();
          if ( y > m_chi2hist->GetYaxis()->GetBinCenter(m_chi2hist->GetYaxis()->GetLast())  ||
                  y < m_chi2hist->GetYaxis()->GetBinCenter(m_chi2hist->GetYaxis()->GetFirst()) ){ // is y value in range of data curve ?
              m_logger << kDEBUG << "posible y=" << y 
                  << " min " << m_chi2hist->GetYaxis()->GetBinCenter(m_chi2hist->GetYaxis()->GetFirst())
                  << " max " << m_chi2hist->GetYaxis()->GetBinCenter(m_chi2hist->GetYaxis()->GetLast())
                  << GEndl;
              retVal = 1000;
          } else {
              if ( fitValue > m_chi2hist->GetXaxis()->GetBinCenter(m_chi2hist->GetXaxis()->GetLast())  ||
                      fitValue < m_chi2hist->GetXaxis()->GetBinCenter(m_chi2hist->GetXaxis()->GetFirst()) ){
                  m_logger << kDEBUG << "imposible x=" << fitValue 
                      << " min " << this->GetMin() << " " << m_chi2hist->GetXaxis()->GetBinCenter(m_chi2hist->GetXaxis()->GetFirst())
                      << " max " << this->GetMax() << " " << m_chi2hist->GetXaxis()->GetBinCenter(m_chi2hist->GetXaxis()->GetLast())
                      << GEndl;
              }
              else {
                  retVal = m_chi2hist->Interpolate(fitValue , y);
              }
          }
      } else {
          retVal=0;
      }

      // possible due to spline
      if (retVal < 0) {
          //m_logger << kWARNING << "<GetChiSquared> Chisq2= "<< retVal << GEndl;
          //retVal = 0.;
      }
      else if (isnan(retVal)) {
          m_logger << kWARNING << "<GetChiSquared> Chisq2 is \"nan\" at " << fitValue 
              << " --> Setting to 1000. !! " << GEndl;

          retVal = 1000.;
      }
  }

  return retVal;
}

Double_t Gfitter::GDataCurve::GetMin() const
{
   return m_chi2hist->GetXaxis()->GetBinCenter(m_chi2hist->GetXaxis()->GetFirst());
}

Double_t Gfitter::GDataCurve::GetMax() const
{
   return m_chi2hist->GetXaxis()->GetBinCenter(m_chi2hist->GetXaxis()->GetLast());
}

Double_t Gfitter::GDataCurve::GetValue() const
{
   Int_t mx,my,mz;
   m_chi2hist->GetMinimumBin(mx,my,mz);
   return m_chi2hist->GetXaxis()->GetBinCenter(mx);
}

Bool_t Gfitter::GDataCurve::SetValue( Double_t ) 
{
   // move minimum of histogram here
   m_logger << kFATAL << "setting of values for GDataCurve still to be implemented" << GEndl;
   return 0;
}

Double_t Gfitter::GDataCurve::GetErrGaussp() const {
   // m_logger << kFATAL << "GetErrGaussp not yet implemented for GDataCurve" << GEndl;
   // MG: Set Error to 0
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrGaussm() const {
   //m_logger << kFATAL << "GetErrGaussm not yet implemented for GDataCurve" << GEndl;
   // MG: Set Error to 0
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrTotp() const {
   m_logger << kFATAL << "GetErrTotp not yet implemented for GDataCurve" << GEndl;
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrTotm() const {
   m_logger << kFATAL << "GetErrTotm not yet implemented for GDataCurve" << GEndl;
   return 0.;
}

Double_t Gfitter::GDataCurve::GetErrTotSym() const {
   m_logger << kFATAL << "GetErrTotSym not yet implemented for GDataCurve" << GEndl;
   return 0.;
}

Double_t Gfitter::GDataCurve::GetRndValue( Double_t mean )
{
   Double_t offset0 = GetValue() - mean;
   Double_t value = 0;

   if(m_chi2hist->GetDimension()==2){
       // set graph as slice of hist at actual y value
       Double_t x = 0;
       GParameter *parF =  gStore()->GetParameter(m_parNameFriend);
       Double_t y = parF->HasTheory() ? parF->GetTheoryPrediction() : parF->GetFitValue();
       for(int i=1; i<=m_chi2hist->GetNbinsX(); i++){
           x = m_chi2hist->GetXaxis()->GetBinCenter(i);
           m_graph->SetPoint(i, x, m_chi2hist->Interpolate(x,y));
       }
   }

   GGraph2PDF *pdf  = new GGraph2PDF(m_graph, offset0);
   TF1 *f1 = new TF1("pdf", pdf, &GGraph2PDF::Evaluate, this->GetMin(), this->GetMax(), 0, "GGraph2PDF", "Evaluate");
   m_logger << kINFO << "Here we are!!!" << GEndl;
   value = f1->GetRandom();

   m_offset =  offset0 + value - mean;
   m_logger << kINFO << offset0 << " + " <<  value << " - " << mean << " = " << m_offset << GEndl;
   delete f1;
   return value;
}

Double_t Gfitter::GDataCurve::GetGaussRnd( Double_t ) const
{
   m_logger << kFATAL << "GetGaussRnd is not defined for GDataCurve" << GEndl;
   return 0.;
}


TH1* Gfitter::GDataCurve::flip_hist_axis(TString histName, TFile *f1) {
    TH1 *hist = (TH1*)f1->Get(histName); 
    if(hist==0) return 0;

    TH2F * flipped = new TH2F( "flipped", "",
           hist->GetYaxis()->GetNbins(),
           hist->GetYaxis()->GetXmin(),
           hist->GetYaxis()->GetXmax(),
           hist->GetXaxis()->GetNbins(),
           hist->GetXaxis()->GetXmin(),
           hist->GetXaxis()->GetXmax()
           );

    for(int xbin=1; xbin < hist->GetNbinsX(); xbin++){
        for(int ybin=1; ybin < hist->GetNbinsY(); ybin++){
            flipped->SetBinContent(ybin,xbin,hist->GetBinContent(xbin,ybin));
        }
    }


    return (TH1*) flipped;
}
